<?php
/**
 * One-Time Setup Script
 * Run this file once to set up the database and webhook
 */

require_once __DIR__ . '/bot/config.php';

echo "<!DOCTYPE html>
<html>
<head>
    <title>Setup - SMM Bot</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #333; }
        .success { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 12px; border-radius: 5px; margin: 10px 0; }
        .error { background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 12px; border-radius: 5px; margin: 10px 0; }
        .info { background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 12px; border-radius: 5px; margin: 10px 0; }
        code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; }
    </style>
</head>
<body>
    <div class='container'>
        <h1>🤖 Telegram SMM Bot Setup</h1>";

// Check if already set up
if (file_exists(__DIR__ . '/.setup_complete')) {
    echo "<div class='error'>⚠️ Setup has already been completed. Delete the file <code>.setup_complete</code> to run setup again.</div>";
    echo "</div></body></html>";
    exit;
}

// Step 1: Test database connection
echo "<h2>Step 1: Database Connection</h2>";
try {
    $db = new PDO(
        "mysql:host=" . DB_HOST . ";charset=utf8mb4",
        DB_USER,
        DB_PASS
    );
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "<div class='success'>✅ Database connection successful</div>";
    
    // Create database if not exists
    $db->exec("CREATE DATABASE IF NOT EXISTS `" . DB_NAME . "` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    echo "<div class='success'>✅ Database created/verified: " . DB_NAME . "</div>";
    
    // Select database
    $db->exec("USE `" . DB_NAME . "`");
    
} catch(PDOException $e) {
    echo "<div class='error'>❌ Database connection failed: " . $e->getMessage() . "</div>";
    echo "<div class='info'>Please check your database credentials in <code>bot/config.php</code></div>";
    echo "</div></body></html>";
    exit;
}

// Step 2: Create tables
echo "<h2>Step 2: Creating Database Tables</h2>";
try {
    $sql = file_get_contents(__DIR__ . '/database.sql');
    
    // Split by semicolon and execute each statement
    $statements = array_filter(array_map('trim', explode(';', $sql)));
    
    foreach ($statements as $statement) {
        if (!empty($statement)) {
            $db->exec($statement);
        }
    }
    
    echo "<div class='success'>✅ All database tables created successfully</div>";
    
} catch(PDOException $e) {
    echo "<div class='error'>❌ Error creating tables: " . $e->getMessage() . "</div>";
    echo "</div></body></html>";
    exit;
}

// Step 3: Set up webhook
echo "<h2>Step 3: Setting Up Telegram Webhook</h2>";

if (BOT_TOKEN === 'YOUR_BOT_TOKEN_HERE') {
    echo "<div class='error'>❌ Please set your bot token in <code>bot/config.php</code></div>";
} else {
    require_once __DIR__ . '/bot/telegram.php';
    
    $result = setWebhook(WEBHOOK_URL);
    
    if ($result['ok']) {
        echo "<div class='success'>✅ Webhook set successfully</div>";
        echo "<div class='info'>Webhook URL: <code>" . WEBHOOK_URL . "</code></div>";
    } else {
        echo "<div class='error'>❌ Webhook setup failed: " . ($result['description'] ?? 'Unknown error') . "</div>";
        echo "<div class='info'>Make sure your webhook URL is accessible via HTTPS</div>";
    }
    
    // Get webhook info
    $info = getWebhookInfo();
    if ($info['ok']) {
        echo "<div class='info'>";
        echo "<strong>Webhook Info:</strong><br>";
        echo "URL: " . ($info['result']['url'] ?? 'Not set') . "<br>";
        echo "Pending updates: " . ($info['result']['pending_update_count'] ?? 0) . "<br>";
        if (isset($info['result']['last_error_message'])) {
            echo "Last error: " . $info['result']['last_error_message'] . "<br>";
        }
        echo "</div>";
    }
}

// Step 4: Verify/Create admin account
echo "<h2>Step 4: Admin Account</h2>";

// Check if admin exists
$stmt = $db->query("SELECT username FROM admin_users WHERE username = 'admin'");
$admin = $stmt->fetch();

if (!$admin) {
    // Create admin account with proper password hash
    echo "<div class='info'>Creating admin account...</div>";
    $password = 'admin123';
    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
    
    try {
        $stmt = $db->prepare("INSERT INTO admin_users (username, password, email) VALUES (?, ?, ?)");
        $stmt->execute(['admin', $hashed_password, 'admin@example.com']);
        echo "<div class='success'>✅ Admin account created successfully</div>";
    } catch(PDOException $e) {
        echo "<div class='error'>❌ Error creating admin account: " . $e->getMessage() . "</div>";
    }
}

if ($admin || $stmt->rowCount() > 0) {
    echo "<div class='success'>✅ Admin account ready</div>";
    echo "<div class='info'>";
    echo "<strong>Admin Login Credentials:</strong><br>";
    echo "Username: <code>admin</code><br>";
    echo "Password: <code>admin123</code><br>";
    echo "<br><strong>⚠️ IMPORTANT:</strong> Change these credentials after first login!";
    echo "</div>";
}

// Step 5: Create states directory
echo "<h2>Step 5: File Permissions</h2>";
$states_dir = __DIR__ . '/bot/states';
if (!is_dir($states_dir)) {
    mkdir($states_dir, 0755, true);
}
if (is_writable($states_dir)) {
    echo "<div class='success'>✅ States directory is writable</div>";
} else {
    echo "<div class='error'>❌ States directory is not writable. Please set permissions to 755</div>";
}

// Mark setup as complete
file_put_contents(__DIR__ . '/.setup_complete', date('Y-m-d H:i:s'));

echo "<h2>🎉 Setup Complete!</h2>";
echo "<div class='success'>";
echo "<strong>Next Steps:</strong><br>";
echo "1. Access admin panel: <a href='admin/'>Click here</a><br>";
echo "2. Login with credentials above<br>";
echo "3. Start your bot on Telegram and send <code>/start</code><br>";
echo "4. Configure service pricing in admin panel<br>";
echo "</div>";

echo "<div class='info'>";
echo "<strong>Important Files:</strong><br>";
echo "• Bot webhook: <code>bot/webhook.php</code><br>";
echo "• Admin panel: <code>admin/</code><br>";
echo "• Configuration: <code>bot/config.php</code><br>";
echo "</div>";

echo "</div></body></html>";
