<?php
/**
 * Telegram API Wrapper Functions
 */

require_once __DIR__ . '/config.php';

function sendTelegramRequest($method, $data = []) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/" . $method;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($result, true);
}

function sendMessage($chat_id, $text, $reply_markup = null, $parse_mode = 'HTML') {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => $parse_mode
    ];
    
    if ($reply_markup) {
        $data['reply_markup'] = $reply_markup;
    }
    
    return sendTelegramRequest('sendMessage', $data);
}

function editMessageText($chat_id, $message_id, $text, $reply_markup = null, $parse_mode = 'HTML') {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => $parse_mode
    ];
    
    if ($reply_markup) {
        $data['reply_markup'] = $reply_markup;
    }
    
    return sendTelegramRequest('editMessageText', $data);
}

function answerCallbackQuery($callback_query_id, $text = '', $show_alert = false) {
    return sendTelegramRequest('answerCallbackQuery', [
        'callback_query_id' => $callback_query_id,
        'text' => $text,
        'show_alert' => $show_alert
    ]);
}

function setWebhook($url) {
    return sendTelegramRequest('setWebhook', ['url' => $url]);
}

function deleteWebhook() {
    return sendTelegramRequest('deleteWebhook');
}

function getWebhookInfo() {
    return sendTelegramRequest('getWebhookInfo');
}

function getChatMember($chat_id, $user_id) {
    return sendTelegramRequest('getChatMember', [
        'chat_id' => $chat_id,
        'user_id' => $user_id
    ]);
}

// Keyboard Builders
function buildInlineKeyboard($buttons) {
    return json_encode(['inline_keyboard' => $buttons]);
}

function buildMainMenu() {
    $buttons = [
        [
            ['text' => '📊 My Balance', 'callback_data' => 'balance'],
            ['text' => '👥 Referrals', 'callback_data' => 'referrals']
        ],
        [
            ['text' => '📱 Instagram Views', 'callback_data' => 'service_instagram_views']
        ],
        [
            ['text' => '👁 Telegram Views', 'callback_data' => 'service_telegram_views'],
            ['text' => '👤 Telegram Members', 'callback_data' => 'service_telegram_members']
        ],
        [
            ['text' => '❤️ Telegram Reactions', 'callback_data' => 'service_telegram_reactions']
        ],
        [
            ['text' => '📜 My Orders', 'callback_data' => 'my_orders']
        ]
    ];
    
    return buildInlineKeyboard($buttons);
}

function buildBackButton() {
    return buildInlineKeyboard([[['text' => '🔙 Back to Menu', 'callback_data' => 'main_menu']]]);
}

function buildConfirmOrderKeyboard($service_type) {
    $buttons = [
        [
            ['text' => '✅ Confirm Order', 'callback_data' => 'confirm_order_' . $service_type],
            ['text' => '❌ Cancel', 'callback_data' => 'main_menu']
        ]
    ];
    
    return buildInlineKeyboard($buttons);
}

// Message Formatters
function formatWelcomeMessage($user) {
    $bot_username = BOT_USERNAME;
    $referral_link = "https://t.me/{$bot_username}?start={$user['referral_code']}";
    
    return "🎉 <b>Welcome to SMM Bot!</b>\n\n" .
           "💰 Your Balance: <b>{$user['points']} points</b>\n\n" .
           "🎁 <b>Earn More Points:</b>\n" .
           "• Refer friends and get " . REFERRAL_POINTS . " points per referral\n" .
           "• Your referral link:\n<code>{$referral_link}</code>\n\n" .
           "📱 <b>Available Services:</b>\n" .
           "• Instagram Views\n" .
           "• Telegram Views\n" .
           "• Telegram Members\n" .
           "• Telegram Reactions\n\n" .
           "💎 Minimum order: " . MIN_ORDER_POINTS . " points\n\n" .
           "Choose a service from the menu below:";
}

function formatBalanceMessage($user, $referral_count) {
    $bot_username = BOT_USERNAME;
    $referral_link = "https://t.me/{$bot_username}?start={$user['referral_code']}";
    
    return "💰 <b>Your Account Balance</b>\n\n" .
           "Points: <b>{$user['points']}</b>\n" .
           "Total Referrals: <b>{$referral_count}</b>\n\n" .
           "🎁 <b>Your Referral Link:</b>\n" .
           "<code>{$referral_link}</code>\n\n" .
           "Share this link to earn " . REFERRAL_POINTS . " points per referral!";
}

function formatServiceMessage($service) {
    return "📱 <b>{$service['service_name']}</b>\n\n" .
           "💎 Price: <b>{$service['points_required']} points</b>\n" .
           "📦 Quantity: <b>{$service['quantity']}</b>\n\n" .
           "Please send the target URL (post/channel link) to place your order.";
}

function formatOrderConfirmation($service, $target_url, $user_points) {
    $remaining = $user_points - $service['points_required'];
    
    return "📋 <b>Order Confirmation</b>\n\n" .
           "Service: <b>{$service['service_name']}</b>\n" .
           "Quantity: <b>{$service['quantity']}</b>\n" .
           "Target: <code>{$target_url}</code>\n\n" .
           "💰 Cost: <b>{$service['points_required']} points</b>\n" .
           "Remaining Balance: <b>{$remaining} points</b>\n\n" .
           "Confirm your order?";
}

function formatOrderSuccess($order_id) {
    return "✅ <b>Order Placed Successfully!</b>\n\n" .
           "Order ID: <b>#{$order_id}</b>\n" .
           "Status: <b>Pending</b>\n\n" .
           "Your order has been sent to admin for processing. " .
           "You will be notified once it's completed!";
}

function formatOrdersList($orders) {
    if (empty($orders)) {
        return "📜 <b>Your Orders</b>\n\nYou haven't placed any orders yet.";
    }
    
    $message = "📜 <b>Your Recent Orders</b>\n\n";
    
    foreach ($orders as $order) {
        $status_emoji = [
            'pending' => '⏳',
            'processing' => '🔄',
            'success' => '✅',
            'failed' => '❌'
        ];
        
        $emoji = $status_emoji[$order['status']] ?? '❓';
        $date = date('d M Y', strtotime($order['created_at']));
        
        $message .= "{$emoji} <b>Order #{$order['id']}</b>\n";
        $message .= "Service: {$order['service_name']}\n";
        $message .= "Points: {$order['points_spent']}\n";
        $message .= "Status: " . ucfirst($order['status']) . "\n";
        $message .= "Date: {$date}\n\n";
    }
    
    return $message;
}

function formatForceJoinMessage($channels) {
    $message = "⚠️ <b>Join Required Channels</b>\n\n";
    $message .= "To use this bot, you must join our channels first:\n\n";
    
    foreach ($channels as $index => $channel) {
        $num = $index + 1;
        $name = $channel['channel_name'] ?: "Channel {$num}";
        $message .= "{$num}. {$name}\n";
    }
    
    $message .= "\nAfter joining all channels, click the button below to verify.";
    
    return $message;
}

function buildForceJoinKeyboard($channels) {
    $buttons = [];
    
    // Add channel buttons
    foreach ($channels as $index => $channel) {
        $num = $index + 1;
        $name = $channel['channel_name'] ?: "Channel {$num}";
        $buttons[] = [
            ['text' => "📢 Join {$name}", 'url' => $channel['channel_link']]
        ];
    }
    
    // Add verification button
    $buttons[] = [
        ['text' => '✅ I Joined All Channels', 'callback_data' => 'verify_join']
    ];
    
    return buildInlineKeyboard($buttons);
}
