<?php
/**
 * Business Logic Handlers
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/telegram.php';

// Store user states in a simple file-based system
function getUserState($telegram_id) {
    $state_file = __DIR__ . '/states/' . $telegram_id . '.json';
    if (file_exists($state_file)) {
        return json_decode(file_get_contents($state_file), true);
    }
    return null;
}

function setUserState($telegram_id, $state) {
    $state_dir = __DIR__ . '/states';
    if (!is_dir($state_dir)) {
        mkdir($state_dir, 0755, true);
    }
    $state_file = $state_dir . '/' . $telegram_id . '.json';
    file_put_contents($state_file, json_encode($state));
}

function clearUserState($telegram_id) {
    $state_file = __DIR__ . '/states/' . $telegram_id . '.json';
    if (file_exists($state_file)) {
        unlink($state_file);
    }
}

// Check if user is member of all required channels
function checkChannelMembership($telegram_id) {
    $db = Database::getInstance()->getConnection();
    
    // Get all active channels
    $stmt = $db->query("SELECT * FROM force_join_channels WHERE is_active = 1");
    $channels = $stmt->fetchAll();
    
    if (empty($channels)) {
        return ['is_member' => true, 'channels' => []];
    }
    
    $not_joined = [];
    
    foreach ($channels as $channel) {
        $result = getChatMember($channel['channel_id'], $telegram_id);
        
        if (!$result || !isset($result['result'])) {
            $not_joined[] = $channel;
            continue;
        }
        
        $status = $result['result']['status'] ?? 'left';
        
        // Check if user is member (member, administrator, creator)
        if (!in_array($status, ['member', 'administrator', 'creator'])) {
            $not_joined[] = $channel;
        }
    }
    
    return [
        'is_member' => empty($not_joined),
        'channels' => $not_joined
    ];
}

// Handle /start command
function handleStart($chat_id, $telegram_id, $username, $first_name, $start_param = null) {
    // Check channel membership first
    $membership = checkChannelMembership($telegram_id);
    
    if (!$membership['is_member']) {
        // User hasn't joined all channels
        $message = formatForceJoinMessage($membership['channels']);
        $keyboard = buildForceJoinKeyboard($membership['channels']);
        sendMessage($chat_id, $message, $keyboard);
        return;
    }
    
    $user = getUserByTelegramId($telegram_id);
    
    if (!$user) {
        // Extract referral code from start parameter
        $referral_code = null;
        if ($start_param && strpos($start_param, 'REF') === 0) {
            $referral_code = $start_param;
        }
        
        // Create new user
        $user = createUser($telegram_id, $username, $first_name, $referral_code);
        
        if ($user) {
            $welcome_text = formatWelcomeMessage($user);
            sendMessage($chat_id, $welcome_text, buildMainMenu());
            
            // Notify referrer if applicable
            if ($referral_code) {
                $referrer = getUserByTelegramId($user['referred_by']);
                if ($referrer) {
                    $notify_text = "🎉 <b>New Referral!</b>\n\n" .
                                  "You earned " . REFERRAL_POINTS . " points!\n" .
                                  "New user: @{$username}\n\n" .
                                  "💰 Your balance: <b>" . ($referrer['points'] + REFERRAL_POINTS) . " points</b>";
                    sendMessage($referrer['telegram_id'], $notify_text);
                }
            }
        } else {
            sendMessage($chat_id, "❌ Error creating account. Please try again later.");
        }
    } else {
        // Existing user
        $welcome_text = formatWelcomeMessage($user);
        sendMessage($chat_id, $welcome_text, buildMainMenu());
    }
    
    clearUserState($telegram_id);
}

// Handle callback queries
function handleCallback($callback_query) {
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $telegram_id = $callback_query['from']['id'];
    $data = $callback_query['data'];
    $callback_id = $callback_query['id'];
    
    // Handle verify join (before user check)
    if ($data === 'verify_join') {
        $membership = checkChannelMembership($telegram_id);
        
        if (!$membership['is_member']) {
            // Still not joined all channels
            answerCallbackQuery($callback_id, "❌ Please join all channels first!", true);
            return;
        }
        
        // User has joined all channels, proceed with normal flow
        $username = $callback_query['from']['username'] ?? '';
        $first_name = $callback_query['from']['first_name'] ?? 'User';
        
        // Delete the force join message
        sendTelegramRequest('deleteMessage', [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ]);
        
        // Send welcome message
        handleStart($chat_id, $telegram_id, $username, $first_name);
        answerCallbackQuery($callback_id, "✅ Verified! Welcome to the bot!");
        return;
    }
    
    $user = getUserByTelegramId($telegram_id);
    
    if (!$user) {
        answerCallbackQuery($callback_id, "Please start the bot first with /start", true);
        return;
    }
    
    // Main menu
    if ($data === 'main_menu') {
        clearUserState($telegram_id);
        $text = formatWelcomeMessage($user);
        editMessageText($chat_id, $message_id, $text, buildMainMenu());
        answerCallbackQuery($callback_id);
        return;
    }
    
    // Balance
    if ($data === 'balance') {
        $referral_count = getReferralCount($user['id']);
        $text = formatBalanceMessage($user, $referral_count);
        editMessageText($chat_id, $message_id, $text, buildBackButton());
        answerCallbackQuery($callback_id);
        return;
    }
    
    // Referrals
    if ($data === 'referrals') {
        $referral_count = getReferralCount($user['id']);
        $bot_username = BOT_USERNAME;
        $referral_link = "https://t.me/{$bot_username}?start={$user['referral_code']}";
        
        $text = "👥 <b>Referral Program</b>\n\n" .
                "Total Referrals: <b>{$referral_count}</b>\n" .
                "Earned: <b>" . ($referral_count * REFERRAL_POINTS) . " points</b>\n\n" .
                "🎁 Earn " . REFERRAL_POINTS . " points for each friend!\n\n" .
                "Your referral link:\n<code>{$referral_link}</code>";
        
        editMessageText($chat_id, $message_id, $text, buildBackButton());
        answerCallbackQuery($callback_id);
        return;
    }
    
    // My Orders
    if ($data === 'my_orders') {
        $orders = getUserOrders($user['id'], 10);
        $text = formatOrdersList($orders);
        editMessageText($chat_id, $message_id, $text, buildBackButton());
        answerCallbackQuery($callback_id);
        return;
    }
    
    // Service selection
    if (strpos($data, 'service_') === 0) {
        $service_type = substr($data, 8);
        $service = getServiceByType($service_type);
        
        if (!$service) {
            answerCallbackQuery($callback_id, "Service not available", true);
            return;
        }
        
        if ($user['points'] < $service['points_required']) {
            $needed = $service['points_required'] - $user['points'];
            answerCallbackQuery($callback_id, "Insufficient points! You need {$needed} more points.", true);
            return;
        }
        
        // Set state to wait for URL
        setUserState($telegram_id, [
            'action' => 'waiting_url',
            'service_type' => $service_type,
            'service_id' => $service['id']
        ]);
        
        $text = formatServiceMessage($service);
        editMessageText($chat_id, $message_id, $text, buildBackButton());
        answerCallbackQuery($callback_id);
        return;
    }
    
    // Confirm order
    if (strpos($data, 'confirm_order_') === 0) {
        $service_type = substr($data, 14);
        $state = getUserState($telegram_id);
        
        if (!$state || !isset($state['target_url'])) {
            answerCallbackQuery($callback_id, "Session expired. Please try again.", true);
            return;
        }
        
        $service = getServiceByType($service_type);
        $target_url = $state['target_url'];
        
        // Create order
        $order_id = createOrder(
            $user['id'],
            $service['id'],
            $service_type,
            $target_url,
            $service['quantity'],
            $service['points_required']
        );
        
        if ($order_id) {
            clearUserState($telegram_id);
            $text = formatOrderSuccess($order_id);
            editMessageText($chat_id, $message_id, $text, buildBackButton());
            answerCallbackQuery($callback_id, "Order placed successfully!");
        } else {
            answerCallbackQuery($callback_id, "Error placing order. Please try again.", true);
        }
        return;
    }
    
    answerCallbackQuery($callback_id);
}

// Handle text messages
function handleMessage($message) {
    $chat_id = $message['chat']['id'];
    $telegram_id = $message['from']['id'];
    $text = $message['text'] ?? '';
    
    $user = getUserByTelegramId($telegram_id);
    
    if (!$user) {
        sendMessage($chat_id, "Please start the bot with /start");
        return;
    }
    
    // Check if waiting for URL
    $state = getUserState($telegram_id);
    
    if ($state && $state['action'] === 'waiting_url') {
        $service_type = $state['service_type'];
        $service = getServiceByType($service_type);
        
        // Basic URL validation
        if (!filter_var($text, FILTER_VALIDATE_URL) && !preg_match('/^https?:\/\//', $text)) {
            sendMessage($chat_id, "❌ Please send a valid URL.\n\nExample:\nhttps://instagram.com/p/xxxxx\nhttps://t.me/channel/123");
            return;
        }
        
        // Update state with URL
        setUserState($telegram_id, [
            'action' => 'confirm_order',
            'service_type' => $service_type,
            'service_id' => $service['id'],
            'target_url' => $text
        ]);
        
        // Show confirmation
        $confirm_text = formatOrderConfirmation($service, $text, $user['points']);
        sendMessage($chat_id, $confirm_text, buildConfirmOrderKeyboard($service_type));
        return;
    }
    
    // Default response
    sendMessage($chat_id, "Please use the menu buttons to interact with the bot.", buildMainMenu());
}
