<?php
/**
 * Database Connection and Helper Functions
 */

require_once __DIR__ . '/config.php';

class Database {
    private static $instance = null;
    private $conn;

    private function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch(PDOException $e) {
            error_log("Database Connection Error: " . $e->getMessage());
            die("Database connection failed");
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->conn;
    }
}

// User Functions
function getUserByTelegramId($telegram_id) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT * FROM users WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    return $stmt->fetch();
}

function createUser($telegram_id, $username, $first_name, $referred_by_code = null) {
    $db = Database::getInstance()->getConnection();
    
    try {
        $db->beginTransaction();
        
        // Generate unique referral code
        $referral_code = generateReferralCode($telegram_id);
        
        // Get referrer ID if referral code provided
        $referred_by_id = null;
        if ($referred_by_code) {
            $stmt = $db->prepare("SELECT id FROM users WHERE referral_code = ?");
            $stmt->execute([$referred_by_code]);
            $referrer = $stmt->fetch();
            if ($referrer) {
                $referred_by_id = $referrer['id'];
            }
        }
        
        // Create user
        $stmt = $db->prepare("
            INSERT INTO users (telegram_id, username, first_name, points, referral_code, referred_by) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $telegram_id, 
            $username, 
            $first_name, 
            STARTING_POINTS, 
            $referral_code,
            $referred_by_id
        ]);
        
        $user_id = $db->lastInsertId();
        
        // Log initial points
        logPointsTransaction($user_id, STARTING_POINTS, 'initial', null, 'Initial signup bonus');
        
        // Award referral points if applicable
        if ($referred_by_id) {
            // Award points to referrer
            $stmt = $db->prepare("UPDATE users SET points = points + ? WHERE id = ?");
            $stmt->execute([REFERRAL_POINTS, $referred_by_id]);
            
            // Log referral
            $stmt = $db->prepare("
                INSERT INTO referrals (referrer_id, referred_id, points_awarded) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute([$referred_by_id, $user_id, REFERRAL_POINTS]);
            
            // Log points transaction
            logPointsTransaction($referred_by_id, REFERRAL_POINTS, 'referral', $user_id, 'Referral bonus');
        }
        
        $db->commit();
        return getUserByTelegramId($telegram_id);
        
    } catch(Exception $e) {
        $db->rollBack();
        error_log("Error creating user: " . $e->getMessage());
        return false;
    }
}

function generateReferralCode($telegram_id) {
    return 'REF' . strtoupper(substr(md5($telegram_id . time()), 0, 8));
}

function updateUserPoints($user_id, $points_change, $type, $reference_id = null, $description = '') {
    $db = Database::getInstance()->getConnection();
    
    try {
        $db->beginTransaction();
        
        $stmt = $db->prepare("UPDATE users SET points = points + ? WHERE id = ?");
        $stmt->execute([$points_change, $user_id]);
        
        logPointsTransaction($user_id, $points_change, $type, $reference_id, $description);
        
        $db->commit();
        return true;
    } catch(Exception $e) {
        $db->rollBack();
        error_log("Error updating points: " . $e->getMessage());
        return false;
    }
}

function logPointsTransaction($user_id, $points_change, $type, $reference_id, $description) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("
        INSERT INTO points_log (user_id, points_change, transaction_type, reference_id, description) 
        VALUES (?, ?, ?, ?, ?)
    ");
    $stmt->execute([$user_id, $points_change, $type, $reference_id, $description]);
}

// Service Functions
function getAllActiveServices() {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM services WHERE is_active = 1 ORDER BY id");
    return $stmt->fetchAll();
}

function getServiceByType($service_type) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT * FROM services WHERE service_type = ? AND is_active = 1");
    $stmt->execute([$service_type]);
    return $stmt->fetch();
}

// Order Functions
function createOrder($user_id, $service_id, $service_type, $target_url, $quantity, $points_spent) {
    $db = Database::getInstance()->getConnection();
    
    try {
        $db->beginTransaction();
        
        // Deduct points
        $stmt = $db->prepare("UPDATE users SET points = points - ? WHERE id = ? AND points >= ?");
        $stmt->execute([$points_spent, $user_id, $points_spent]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("Insufficient points");
        }
        
        // Create order
        $stmt = $db->prepare("
            INSERT INTO orders (user_id, service_id, service_type, target_url, quantity, points_spent, status) 
            VALUES (?, ?, ?, ?, ?, ?, 'pending')
        ");
        $stmt->execute([$user_id, $service_id, $service_type, $target_url, $quantity, $points_spent]);
        
        $order_id = $db->lastInsertId();
        
        // Log points transaction
        logPointsTransaction($user_id, -$points_spent, 'order', $order_id, "Order for $service_type");
        
        $db->commit();
        return $order_id;
        
    } catch(Exception $e) {
        $db->rollBack();
        error_log("Error creating order: " . $e->getMessage());
        return false;
    }
}

function getUserOrders($user_id, $limit = 10) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("
        SELECT o.*, s.service_name 
        FROM orders o 
        JOIN services s ON o.service_id = s.id 
        WHERE o.user_id = ? 
        ORDER BY o.created_at DESC 
        LIMIT ?
    ");
    $stmt->execute([$user_id, $limit]);
    return $stmt->fetchAll();
}

function getReferralCount($user_id) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM referrals WHERE referrer_id = ?");
    $stmt->execute([$user_id]);
    $result = $stmt->fetch();
    return $result['count'];
}
