<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'User Management';
$db = Database::getInstance()->getConnection();

$success = '';
$error = '';

// Handle points update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_points'])) {
    $user_id = $_POST['user_id'];
    $new_points = intval($_POST['points']);
    
    // Get current points
    $stmt = $db->prepare("SELECT points FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $current_points = $stmt->fetch()['points'];
    
    $stmt = $db->prepare("UPDATE users SET points = ? WHERE id = ?");
    if ($stmt->execute([$new_points, $user_id])) {
        // Log the adjustment
        require_once __DIR__ . '/../bot/database.php';
        $diff = $new_points - $current_points;
        logPointsTransaction($user_id, $diff, 'admin_adjustment', null, 'Admin manual adjustment');
        
        $success = "Points updated successfully!";
    }
}

// Get all users with stats
$stmt = $db->query("
    SELECT u.*, 
           (SELECT COUNT(*) FROM orders WHERE user_id = u.id) as total_orders,
           (SELECT COUNT(*) FROM referrals WHERE referrer_id = u.id) as total_referrals,
           (SELECT SUM(points_spent) FROM orders WHERE user_id = u.id) as total_spent
    FROM users u 
    ORDER BY u.created_at DESC
");
$users = $stmt->fetchAll();

// Get total stats
$total_users = count($users);
$total_points = array_sum(array_column($users, 'points'));
$total_referrals = array_sum(array_column($users, 'total_referrals'));

include 'includes/header.php';
?>

<style>
/* Users Page Specific Styles */
.users-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: slideDown 0.6s ease-out;
}

.users-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.users-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

.users-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
}

.user-stat-item {
    display: flex;
    align-items: center;
    gap: 12px;
}

.user-stat-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.user-stat-info {
    display: flex;
    flex-direction: column;
}

.user-stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
}

.user-stat-value {
    font-size: 24px;
    font-weight: 700;
    color: var(--text-primary);
}

/* Users Grid */
.users-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 24px;
}

.user-card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 24px;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    animation: fadeIn 0.6s ease-out;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.user-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-bg);
    transform: scaleX(0);
    transition: transform 0.4s ease;
}

.user-card:hover::before {
    transform: scaleX(1);
}

.user-card:hover {
    transform: translateY(-8px);
    box-shadow: 
        12px 12px 24px var(--shadow-dark),
        -12px -12px 24px var(--shadow-light),
        0 0 40px rgba(99, 102, 241, 0.2);
}

.user-card-header {
    display: flex;
    align-items: center;
    gap: 16px;
    margin-bottom: 20px;
    padding-bottom: 20px;
    border-bottom: 1px solid var(--border-color);
}

.user-avatar {
    width: 64px;
    height: 64px;
    border-radius: 50%;
    background: var(--gradient-bg);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 28px;
    font-weight: 700;
    flex-shrink: 0;
    box-shadow: 0 8px 16px rgba(99, 102, 241, 0.3);
    animation: float 3s ease-in-out infinite;
}

.user-info {
    flex: 1;
    min-width: 0;
}

.user-name {
    font-size: 20px;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 4px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.user-username {
    font-size: 14px;
    color: var(--text-secondary);
    font-family: 'Courier New', monospace;
}

.user-id-badge {
    padding: 4px 10px;
    background: var(--bg-secondary);
    border-radius: 12px;
    font-size: 11px;
    color: var(--text-secondary);
    font-weight: 600;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.user-stats-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 12px;
    margin-bottom: 20px;
}

.user-stat-box {
    text-align: center;
    padding: 12px;
    background: var(--bg-secondary);
    border-radius: 12px;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.user-stat-box-label {
    font-size: 10px;
    color: var(--text-secondary);
    text-transform: uppercase;
    margin-bottom: 4px;
}

.user-stat-box-value {
    font-size: 20px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.user-referral-code {
    padding: 12px;
    background: var(--bg-secondary);
    border-radius: 10px;
    margin-bottom: 16px;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.user-referral-code-label {
    font-size: 11px;
    color: var(--text-secondary);
    margin-bottom: 6px;
}

.user-referral-code-value {
    font-family: 'Courier New', monospace;
    font-size: 14px;
    color: var(--primary-light);
    font-weight: 600;
}

.user-points-form {
    display: flex;
    gap: 8px;
    margin-bottom: 12px;
}

.user-points-input {
    flex: 1;
    padding: 10px 14px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 10px;
    color: var(--text-primary);
    font-size: 16px;
    font-weight: 700;
    transition: all 0.3s ease;
}

.user-points-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.user-actions {
    display: flex;
    gap: 8px;
}

.user-action-btn {
    flex: 1;
    padding: 10px;
    border: none;
    border-radius: 10px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 
        4px 4px 8px var(--shadow-dark),
        -4px -4px 8px var(--shadow-light);
}

.user-action-btn.primary {
    background: rgba(99, 102, 241, 0.2);
    color: var(--primary-light);
    border: 1px solid rgba(99, 102, 241, 0.3);
}

.user-action-btn.primary:hover {
    background: rgba(99, 102, 241, 0.3);
    transform: translateY(-2px);
}

.user-action-btn.success {
    background: rgba(16, 185, 129, 0.2);
    color: #6ee7b7;
    border: 1px solid rgba(16, 185, 129, 0.3);
}

.user-action-btn.success:hover {
    background: rgba(16, 185, 129, 0.3);
    transform: translateY(-2px);
}

.user-joined {
    font-size: 12px;
    color: var(--text-secondary);
    text-align: center;
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid var(--border-color);
}

/* User Details Modal */
.user-details-modal {
    margin-bottom: 24px;
}

.user-details-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
}

.user-details-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.user-detail-stat {
    padding: 20px;
    background: var(--bg-secondary);
    border-radius: 16px;
    text-align: center;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.user-detail-stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    margin-bottom: 8px;
}

.user-detail-stat-value {
    font-size: 32px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 16px;
    padding-bottom: 12px;
    border-bottom: 2px solid var(--border-color);
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes float {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-6px); }
}
</style>

<!-- Users Header -->
<div class="users-header">
    <div class="users-title">
        <h1>👥 User Management</h1>
    </div>
    
    <div class="users-stats">
        <div class="user-stat-item">
            <div class="user-stat-icon" style="background: rgba(99, 102, 241, 0.1);">👤</div>
            <div class="user-stat-info">
                <span class="user-stat-label">Total Users</span>
                <span class="user-stat-value"><?php echo $total_users; ?></span>
            </div>
        </div>
        <div class="user-stat-item">
            <div class="user-stat-icon" style="background: rgba(16, 185, 129, 0.1);">💎</div>
            <div class="user-stat-info">
                <span class="user-stat-label">Total Points</span>
                <span class="user-stat-value"><?php echo number_format($total_points); ?></span>
            </div>
        </div>
        <div class="user-stat-item">
            <div class="user-stat-icon" style="background: rgba(139, 92, 246, 0.1);">🔗</div>
            <div class="user-stat-info">
                <span class="user-stat-label">Total Referrals</span>
                <span class="user-stat-value"><?php echo $total_referrals; ?></span>
            </div>
        </div>
    </div>
</div>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<?php if (isset($_GET['view_user'])): ?>
    <?php
    $user_id = $_GET['view_user'];
    
    // Get user details
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user_detail = $stmt->fetch();
    
    if ($user_detail):
        // Get user orders
        $stmt = $db->prepare("
            SELECT o.*, s.service_name 
            FROM orders o 
            JOIN services s ON o.service_id = s.id 
            WHERE o.user_id = ? 
            ORDER BY o.created_at DESC 
            LIMIT 20
        ");
        $stmt->execute([$user_id]);
        $user_orders = $stmt->fetchAll();
        
        // Get referrals
        $stmt = $db->prepare("
            SELECT u.* 
            FROM users u 
            JOIN referrals r ON u.id = r.referred_id 
            WHERE r.referrer_id = ?
        ");
        $stmt->execute([$user_id]);
        $referrals = $stmt->fetchAll();
        
        $total_spent = array_sum(array_column($user_orders, 'points_spent'));
    ?>

    <div class="card user-details-modal">
        <div class="card-header user-details-header">
            <h2 class="card-title">👤 <?php echo htmlspecialchars($user_detail['first_name']); ?></h2>
            <a href="users.php" class="btn btn-primary btn-sm">← Back to List</a>
        </div>
        
        <div style="padding: 0 10px 20px;">
            <div class="user-details-stats">
                <div class="user-detail-stat">
                    <div class="user-detail-stat-label">Telegram ID</div>
                    <div class="user-detail-stat-value" style="font-size: 20px; font-family: 'Courier New', monospace;">
                        <?php echo $user_detail['telegram_id']; ?>
                    </div>
                </div>
                <div class="user-detail-stat">
                    <div class="user-detail-stat-label">Points Balance</div>
                    <div class="user-detail-stat-value"><?php echo number_format($user_detail['points']); ?></div>
                </div>
                <div class="user-detail-stat">
                    <div class="user-detail-stat-label">Total Referrals</div>
                    <div class="user-detail-stat-value"><?php echo count($referrals); ?></div>
                </div>
                <div class="user-detail-stat">
                    <div class="user-detail-stat-label">Total Orders</div>
                    <div class="user-detail-stat-value"><?php echo count($user_orders); ?></div>
                </div>
                <div class="user-detail-stat">
                    <div class="user-detail-stat-label">Points Spent</div>
                    <div class="user-detail-stat-value"><?php echo number_format($total_spent); ?></div>
                </div>
            </div>
            
            <h3 class="section-title">📋 Recent Orders</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Order ID</th>
                            <th>Service</th>
                            <th>Points</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($user_orders)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center; color: var(--text-secondary);">No orders yet</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($user_orders as $order): ?>
                                <tr>
                                    <td><strong>#<?php echo $order['id']; ?></strong></td>
                                    <td><?php echo htmlspecialchars($order['service_name']); ?></td>
                                    <td><strong><?php echo number_format($order['points_spent']); ?></strong></td>
                                    <td><span class="badge badge-<?php echo $order['status']; ?>"><?php echo ucfirst($order['status']); ?></span></td>
                                    <td><?php echo date('d M Y, H:i', strtotime($order['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if (!empty($referrals)): ?>
                <h3 class="section-title" style="margin-top: 30px;">🔗 Referrals (<?php echo count($referrals); ?>)</h3>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Username</th>
                                <th>Points</th>
                                <th>Joined</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($referrals as $ref): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($ref['first_name']); ?></td>
                                    <td>
                                        <?php if ($ref['username']): ?>
                                            @<?php echo htmlspecialchars($ref['username']); ?>
                                        <?php else: ?>
                                            <span style="color: var(--text-secondary);">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><strong><?php echo number_format($ref['points']); ?></strong></td>
                                    <td><?php echo date('d M Y', strtotime($ref['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php endif; ?>
<?php else: ?>

    <!-- Users Grid -->
    <div class="users-grid">
        <?php if (empty($users)): ?>
            <div class="card">
                <div class="empty-state">
                    <div class="empty-state-icon">👥</div>
                    <div class="empty-state-text">No users yet</div>
                    <p style="margin-top: 8px;">Users will appear here when they start using the bot</p>
                </div>
            </div>
        <?php else: ?>
            <?php foreach ($users as $user): ?>
                <div class="user-card">
                    <div class="user-card-header">
                        <div class="user-avatar">
                            <?php echo strtoupper(substr($user['first_name'], 0, 1)); ?>
                        </div>
                        <div class="user-info">
                            <div class="user-name"><?php echo htmlspecialchars($user['first_name']); ?></div>
                            <div class="user-username">
                                <?php if ($user['username']): ?>
                                    @<?php echo htmlspecialchars($user['username']); ?>
                                <?php else: ?>
                                    No username
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="user-id-badge">ID: <?php echo $user['id']; ?></div>
                    </div>
                    
                    <div class="user-stats-grid">
                        <div class="user-stat-box">
                            <div class="user-stat-box-label">Points</div>
                            <div class="user-stat-box-value"><?php echo number_format($user['points']); ?></div>
                        </div>
                        <div class="user-stat-box">
                            <div class="user-stat-box-label">Referrals</div>
                            <div class="user-stat-box-value"><?php echo $user['total_referrals']; ?></div>
                        </div>
                        <div class="user-stat-box">
                            <div class="user-stat-box-label">Orders</div>
                            <div class="user-stat-box-value"><?php echo $user['total_orders']; ?></div>
                        </div>
                    </div>
                    
                    <div class="user-referral-code">
                        <div class="user-referral-code-label">Referral Code</div>
                        <div class="user-referral-code-value"><?php echo $user['referral_code']; ?></div>
                    </div>
                    
                    <form method="POST" class="user-points-form">
                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                        <input type="number" name="points" value="<?php echo $user['points']; ?>" 
                               class="user-points-input" min="0" step="1">
                        <button type="submit" name="update_points" class="user-action-btn success">
                            💾 Save
                        </button>
                    </form>
                    
                    <div class="user-actions">
                        <a href="?view_user=<?php echo $user['id']; ?>" class="user-action-btn primary">
                            👁️ View Details
                        </a>
                    </div>
                    
                    <div class="user-joined">
                        📅 Joined: <?php echo date('d M Y', strtotime($user['created_at'])); ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

<?php endif; ?>

<?php include 'includes/footer.php'; ?>
