<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'Service Management';
$db = Database::getInstance()->getConnection();

$success = '';
$error = '';

// Handle service add
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_service'])) {
    $service_name = trim($_POST['service_name']);
    $service_type = strtolower(str_replace(' ', '_', trim($_POST['service_type'])));
    $points_required = intval($_POST['points_required']);
    $quantity = intval($_POST['quantity']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    if (empty($service_name) || empty($service_type)) {
        $error = "Service name and type are required";
    } else {
        try {
            $stmt = $db->prepare("INSERT INTO services (service_name, service_type, points_required, quantity, is_active) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$service_name, $service_type, $points_required, $quantity, $is_active]);
            $success = "Service added successfully!";
        } catch(PDOException $e) {
            if ($e->getCode() == 23000) {
                $error = "Service type already exists!";
            } else {
                $error = "Error adding service: " . $e->getMessage();
            }
        }
    }
}

// Handle service update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_service'])) {
    $service_id = $_POST['service_id'];
    $service_name = trim($_POST['service_name']);
    $points_required = intval($_POST['points_required']);
    $quantity = intval($_POST['quantity']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    $stmt = $db->prepare("UPDATE services SET service_name = ?, points_required = ?, quantity = ?, is_active = ? WHERE id = ?");
    if ($stmt->execute([$service_name, $points_required, $quantity, $is_active, $service_id])) {
        $success = "Service updated successfully!";
    }
}

// Handle service delete
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $service_id = $_GET['delete'];
    
    try {
        // Check if service has orders
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM orders WHERE service_id = ?");
        $stmt->execute([$service_id]);
        $count = $stmt->fetch()['count'];
        
        if ($count > 0) {
            $error = "Cannot delete service with existing orders. Deactivate it instead.";
        } else {
            $stmt = $db->prepare("DELETE FROM services WHERE id = ?");
            $stmt->execute([$service_id]);
            $success = "Service deleted successfully!";
        }
    } catch(PDOException $e) {
        $error = "Error deleting service: " . $e->getMessage();
    }
}

// Get all services
$stmt = $db->query("SELECT s.*, COUNT(o.id) as order_count FROM services s LEFT JOIN orders o ON s.id = o.service_id GROUP BY s.id ORDER BY s.id");
$services = $stmt->fetchAll();

include 'includes/header.php';
?>

<style>
/* Services Page Specific Styles */
.services-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: slideDown 0.6s ease-out;
}

.services-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.services-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

.add-service-btn {
    padding: 12px 24px;
    background: var(--gradient-bg);
    border: none;
    border-radius: 12px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
    display: flex;
    align-items: center;
    gap: 8px;
}

.add-service-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(99, 102, 241, 0.4);
}

.add-service-form {
    margin-bottom: 30px;
    animation: slideDown 0.4s ease-out;
}

.btn-close {
    background: none;
    border: none;
    color: var(--text-secondary);
    font-size: 24px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-close:hover {
    color: var(--danger);
    transform: rotate(90deg);
}

.service-form {
    padding: 10px 0;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.form-group-modern {
    margin-bottom: 20px;
}

.form-group-modern label {
    display: block;
    margin-bottom: 8px;
    color: var(--text-primary);
    font-weight: 500;
    font-size: 14px;
}

.form-group-modern input[type="text"],
.form-group-modern input[type="number"] {
    width: 100%;
    padding: 12px 16px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    color: var(--text-primary);
    font-size: 15px;
    transition: all 0.3s ease;
}

.form-group-modern input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 4px rgba(99, 102, 241, 0.1);
}

.form-group-modern small {
    display: block;
    margin-top: 6px;
    color: var(--text-secondary);
    font-size: 12px;
}

.toggle-switch {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
}

.toggle-switch input[type="checkbox"] {
    display: none;
}

.toggle-slider {
    position: relative;
    width: 50px;
    height: 26px;
    background: rgba(51, 65, 85, 0.6);
    border-radius: 13px;
    transition: all 0.3s ease;
    box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.2);
}

.toggle-slider::before {
    content: '';
    position: absolute;
    width: 22px;
    height: 22px;
    border-radius: 50%;
    background: white;
    top: 2px;
    left: 2px;
    transition: all 0.3s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
}

.toggle-switch input:checked + .toggle-slider {
    background: var(--gradient-4);
}

.toggle-switch input:checked + .toggle-slider::before {
    transform: translateX(24px);
}

.toggle-label {
    color: var(--text-primary);
    font-size: 14px;
}

.form-actions {
    display: flex;
    gap: 12px;
    margin-top: 24px;
}

.btn-secondary {
    background: rgba(51, 65, 85, 0.6);
    color: var(--text-primary);
}

.btn-secondary:hover {
    background: rgba(51, 65, 85, 0.8);
}

/* Pricing Guide */
.pricing-guide {
    margin-bottom: 30px;
}

.guide-content {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.guide-item {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    padding: 16px;
    background: rgba(99, 102, 241, 0.05);
    border-radius: 12px;
    border: 1px solid rgba(99, 102, 241, 0.1);
    transition: all 0.3s ease;
}

.guide-item:hover {
    background: rgba(99, 102, 241, 0.08);
    transform: translateY(-2px);
}

.guide-icon {
    font-size: 24px;
    flex-shrink: 0;
}

.guide-text {
    font-size: 14px;
    line-height: 1.6;
    color: var(--text-primary);
}

.guide-text strong {
    display: block;
    margin-bottom: 4px;
    color: var(--primary-light);
}

/* Services Grid */
.services-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
    gap: 24px;
}

.service-card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 24px;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    animation: fadeIn 0.6s ease-out;
    
    /* Neumorphism effect */
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.service-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-bg);
    transform: scaleX(0);
    transition: transform 0.4s ease;
}

.service-card:hover::before {
    transform: scaleX(1);
}

.service-card:hover {
    transform: translateY(-8px);
    border-color: rgba(99, 102, 241, 0.4);
    box-shadow: 
        12px 12px 24px var(--shadow-dark),
        -12px -12px 24px var(--shadow-light),
        0 0 40px rgba(99, 102, 241, 0.2);
}

.service-card.inactive {
    opacity: 0.6;
}

.service-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
}

.service-icon {
    font-size: 40px;
    width: 70px;
    height: 70px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: rgba(99, 102, 241, 0.1);
    border-radius: 16px;
    animation: float 3s ease-in-out infinite;
    
    /* Neumorphism */
    box-shadow: 
        inset 4px 4px 8px var(--shadow-dark),
        inset -4px -4px 8px var(--shadow-light);
}

@keyframes float {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-8px); }
}

.status-badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
}

.status-badge.active {
    background: rgba(16, 185, 129, 0.15);
    color: #6ee7b7;
    border: 1px solid rgba(16, 185, 129, 0.3);
}

.status-badge.inactive {
    background: rgba(148, 163, 184, 0.15);
    color: var(--text-secondary);
    border: 1px solid rgba(148, 163, 184, 0.3);
}

.service-name {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 6px;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.service-type {
    color: var(--text-secondary);
    font-size: 13px;
    margin-bottom: 20px;
    font-family: 'Courier New', monospace;
}

.service-stats {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px;
    background: var(--bg-secondary);
    border-radius: 12px;
    margin-bottom: 20px;
    
    /* Neumorphism inset */
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.stat-item {
    text-align: center;
}

.stat-label {
    font-size: 11px;
    color: var(--text-secondary);
    text-transform: uppercase;
    margin-bottom: 4px;
}

.stat-value {
    font-size: 18px;
    font-weight: 700;
    color: var(--primary-light);
}

.stat-divider {
    width: 1px;
    height: 30px;
    background: var(--border-color);
}

.service-actions {
    display: flex;
    gap: 10px;
}

.btn-action {
    flex: 1;
    padding: 10px;
    border: none;
    border-radius: 10px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    
    /* Neumorphism button */
    box-shadow: 
        4px 4px 8px var(--shadow-dark),
        -4px -4px 8px var(--shadow-light);
}

.btn-action.edit {
    background: rgba(59, 130, 246, 0.2);
    color: #93c5fd;
    border: 1px solid rgba(59, 130, 246, 0.3);
}

.btn-action.edit:hover {
    background: rgba(59, 130, 246, 0.3);
    transform: translateY(-2px);
    box-shadow: 
        6px 6px 12px var(--shadow-dark),
        -6px -6px 12px var(--shadow-light);
}

.btn-action.delete {
    background: rgba(239, 68, 68, 0.2);
    color: #fca5a5;
    border: 1px solid rgba(239, 68, 68, 0.3);
}

.btn-action.delete:hover {
    background: rgba(239, 68, 68, 0.3);
    transform: translateY(-2px);
    box-shadow: 
        6px 6px 12px var(--shadow-dark),
        -6px -6px 12px var(--shadow-light);
}

.btn-action.disabled {
    background: rgba(51, 65, 85, 0.2);
    color: var(--text-secondary);
    border: 1px solid rgba(51, 65, 85, 0.3);
    cursor: not-allowed;
    opacity: 0.5;
}

.edit-form {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
    animation: slideDown 0.3s ease-out;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>

<!-- Services Header -->
<div class="services-header">
    <div class="services-title">
        <h1>⚙️ Service Management</h1>
        <button class="add-service-btn" onclick="toggleAddForm()">
            ➕ Add New Service
        </button>
    </div>
</div>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<!-- Add Service Form (Hidden by default) -->
<div class="card add-service-form" id="addServiceForm" style="display: none;">
    <div class="card-header">
        <h2 class="card-title">➕ Add New Service</h2>
        <button class="btn-close" onclick="toggleAddForm()">✕</button>
    </div>
    
    <form method="POST" class="service-form">
        <div class="form-grid">
            <div class="form-group-modern">
                <label for="service_name">Service Name</label>
                <input type="text" id="service_name" name="service_name" placeholder="e.g., YouTube Views" required>
            </div>
            
            <div class="form-group-modern">
                <label for="service_type">Service Type (Unique ID)</label>
                <input type="text" id="service_type" name="service_type" placeholder="e.g., youtube_views" required>
                <small>Use lowercase with underscores, no spaces</small>
            </div>
            
            <div class="form-group-modern">
                <label for="points_required">Points Required</label>
                <input type="number" id="points_required" name="points_required" value="400" min="1" required>
            </div>
            
            <div class="form-group-modern">
                <label for="quantity">Quantity Delivered</label>
                <input type="number" id="quantity" name="quantity" value="1000" min="1" required>
            </div>
        </div>
        
        <div class="form-group-modern">
            <label class="toggle-switch">
                <input type="checkbox" name="is_active" checked>
                <span class="toggle-slider"></span>
                <span class="toggle-label">Active (Users can order this service)</span>
            </label>
        </div>
        
        <div class="form-actions">
            <button type="submit" name="add_service" class="btn btn-primary">
                ✨ Create Service
            </button>
            <button type="button" class="btn btn-secondary" onclick="toggleAddForm()">
                Cancel
            </button>
        </div>
    </form>
</div>

<!-- Pricing Guide -->
<div class="card pricing-guide">
    <div class="card-header">
        <h2 class="card-title">💡 Pricing Guide</h2>
    </div>
    <div class="guide-content">
        <div class="guide-item">
            <div class="guide-icon">📊</div>
            <div class="guide-text">
                <strong>Points Required:</strong> How many points users spend per order
            </div>
        </div>
        <div class="guide-item">
            <div class="guide-icon">📦</div>
            <div class="guide-text">
                <strong>Quantity:</strong> What users receive (e.g., 1000 views, 500 likes)
            </div>
        </div>
        <div class="guide-item">
            <div class="guide-icon">💎</div>
            <div class="guide-text">
                <strong>Example:</strong> 400 points = 1000 views means users pay 400 points to get 1000 views
            </div>
        </div>
        <div class="guide-item">
            <div class="guide-icon">⚙️</div>
            <div class="guide-text">
                <strong>System Settings:</strong> New users get <?php echo STARTING_POINTS; ?> points, Referrals earn <?php echo REFERRAL_POINTS; ?> points
            </div>
        </div>
    </div>
</div>

<!-- Services List -->
<div class="services-grid">
    <?php foreach ($services as $service): ?>
        <div class="service-card <?php echo $service['is_active'] ? 'active' : 'inactive'; ?>">
            <div class="service-header">
                <div class="service-icon">
                    <?php
                    $icons = [
                        'instagram' => '📱',
                        'telegram' => '✈️',
                        'youtube' => '▶️',
                        'facebook' => '👥',
                        'twitter' => '🐦',
                        'tiktok' => '🎵'
                    ];
                    $icon = '⭐';
                    foreach ($icons as $key => $val) {
                        if (strpos($service['service_type'], $key) !== false) {
                            $icon = $val;
                            break;
                        }
                    }
                    echo $icon;
                    ?>
                </div>
                <div class="service-status">
                    <?php if ($service['is_active']): ?>
                        <span class="status-badge active">● Active</span>
                    <?php else: ?>
                        <span class="status-badge inactive">● Inactive</span>
                    <?php endif; ?>
                </div>
            </div>
            
            <h3 class="service-name"><?php echo htmlspecialchars($service['service_name']); ?></h3>
            <p class="service-type"><?php echo htmlspecialchars($service['service_type']); ?></p>
            
            <div class="service-stats">
                <div class="stat-item">
                    <div class="stat-label">Points</div>
                    <div class="stat-value"><?php echo number_format($service['points_required']); ?></div>
                </div>
                <div class="stat-divider"></div>
                <div class="stat-item">
                    <div class="stat-label">Quantity</div>
                    <div class="stat-value"><?php echo number_format($service['quantity']); ?></div>
                </div>
                <div class="stat-divider"></div>
                <div class="stat-item">
                    <div class="stat-label">Orders</div>
                    <div class="stat-value"><?php echo number_format($service['order_count']); ?></div>
                </div>
            </div>
            
            <div class="service-actions">
                <button class="btn-action edit" onclick="editService(<?php echo $service['id']; ?>)">
                    ✏️ Edit
                </button>
                <?php if ($service['order_count'] == 0): ?>
                    <button class="btn-action delete" onclick="deleteService(<?php echo $service['id']; ?>, '<?php echo htmlspecialchars($service['service_name']); ?>')">
                        🗑️ Delete
                    </button>
                <?php else: ?>
                    <button class="btn-action disabled" disabled title="Cannot delete service with orders">
                        🔒 Has Orders
                    </button>
                <?php endif; ?>
            </div>
            
            <!-- Edit Form (Hidden) -->
            <div class="edit-form" id="editForm<?php echo $service['id']; ?>" style="display: none;">
                <form method="POST">
                    <input type="hidden" name="service_id" value="<?php echo $service['id']; ?>">
                    
                    <div class="form-group-modern">
                        <label>Service Name</label>
                        <input type="text" name="service_name" value="<?php echo htmlspecialchars($service['service_name']); ?>" required>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group-modern">
                            <label>Points</label>
                            <input type="number" name="points_required" value="<?php echo $service['points_required']; ?>" min="1" required>
                        </div>
                        <div class="form-group-modern">
                            <label>Quantity</label>
                            <input type="number" name="quantity" value="<?php echo $service['quantity']; ?>" min="1" required>
                        </div>
                    </div>
                    
                    <div class="form-group-modern">
                        <label class="toggle-switch">
                            <input type="checkbox" name="is_active" <?php echo $service['is_active'] ? 'checked' : ''; ?>>
                            <span class="toggle-slider"></span>
                            <span class="toggle-label">Active</span>
                        </label>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" name="update_service" class="btn btn-success btn-sm">
                            💾 Save
                        </button>
                        <button type="button" class="btn btn-secondary btn-sm" onclick="cancelEdit(<?php echo $service['id']; ?>)">
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<script>
function toggleAddForm() {
    const form = document.getElementById('addServiceForm');
    if (form.style.display === 'none') {
        form.style.display = 'block';
        form.scrollIntoView({ behavior: 'smooth' });
    } else {
        form.style.display = 'none';
    }
}

function editService(id) {
    const editForm = document.getElementById('editForm' + id);
    const allForms = document.querySelectorAll('.edit-form');
    
    allForms.forEach(form => {
        if (form.id !== 'editForm' + id) {
            form.style.display = 'none';
        }
    });
    
    if (editForm.style.display === 'none') {
        editForm.style.display = 'block';
    } else {
        editForm.style.display = 'none';
    }
}

function cancelEdit(id) {
    document.getElementById('editForm' + id).style.display = 'none';
}

function deleteService(id, name) {
    if (confirm('Are you sure you want to delete "' + name + '"?\n\nThis action cannot be undone!')) {
        window.location.href = 'services.php?delete=' + id;
    }
}
</script>

<?php include 'includes/footer.php'; ?>
