<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'Order Management';
$db = Database::getInstance()->getConnection();

$success = '';
$error = '';

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    $order_id = $_POST['order_id'];
    $new_status = $_POST['status'];
    $admin_note = $_POST['admin_note'] ?? '';
    
    $stmt = $db->prepare("UPDATE orders SET status = ?, admin_note = ? WHERE id = ?");
    if ($stmt->execute([$new_status, $admin_note, $order_id])) {
        $success = "Order #{$order_id} updated to {$new_status}";
        
        // Notify user via Telegram
        $stmt = $db->prepare("
            SELECT o.*, u.telegram_id, s.service_name 
            FROM orders o 
            JOIN users u ON o.user_id = u.id 
            JOIN services s ON o.service_id = s.id 
            WHERE o.id = ?
        ");
        $stmt->execute([$order_id]);
        $order = $stmt->fetch();
        
        if ($order) {
            require_once __DIR__ . '/../bot/telegram.php';
            
            $status_emoji = [
                'success' => '✅',
                'failed' => '❌',
                'processing' => '🔄'
            ];
            
            $emoji = $status_emoji[$new_status] ?? '📋';
            $message = "{$emoji} <b>Order Update</b>\n\n" .
                      "Order ID: <b>#{$order_id}</b>\n" .
                      "Service: {$order['service_name']}\n" .
                      "Status: <b>" . ucfirst($new_status) . "</b>\n";
            
            if ($admin_note) {
                $message .= "\nNote: {$admin_note}";
            }
            
            sendMessage($order['telegram_id'], $message);
        }
    }
}

// Get filter
$status_filter = $_GET['status'] ?? 'all';

// Get statistics
$stats = [
    'all' => 0,
    'pending' => 0,
    'processing' => 0,
    'success' => 0,
    'failed' => 0
];

$stmt = $db->query("SELECT status, COUNT(*) as count FROM orders GROUP BY status");
while ($row = $stmt->fetch()) {
    $stats[$row['status']] = $row['count'];
    $stats['all'] += $row['count'];
}

// Build query
$query = "
    SELECT o.*, u.username, u.first_name, u.telegram_id, s.service_name 
    FROM orders o 
    JOIN users u ON o.user_id = u.id 
    JOIN services s ON o.service_id = s.id 
";

if ($status_filter !== 'all') {
    $query .= " WHERE o.status = :status";
}

$query .= " ORDER BY o.created_at DESC LIMIT 100";

$stmt = $db->prepare($query);
if ($status_filter !== 'all') {
    $stmt->bindParam(':status', $status_filter);
}
$stmt->execute();
$orders = $stmt->fetchAll();

include 'includes/header.php';
?>

<style>
/* Orders Page Specific Styles */
.orders-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.orders-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.orders-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

/* Filter Tabs */
.filter-tabs {
    display: flex;
    gap: 12px;
    margin-top: 24px;
    padding: 8px;
    background: var(--bg-secondary);
    border-radius: 16px;
    overflow-x: auto;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.filter-tab {
    padding: 12px 24px;
    border-radius: 12px;
    border: none;
    background: transparent;
    color: var(--text-secondary);
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    white-space: nowrap;
    display: flex;
    align-items: center;
    gap: 8px;
    text-decoration: none;
}

.filter-tab:hover {
    background: rgba(99, 102, 241, 0.1);
    color: var(--text-primary);
}

.filter-tab.active {
    background: var(--gradient-bg);
    color: white;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
}

.filter-count {
    padding: 2px 8px;
    border-radius: 10px;
    background: rgba(255, 255, 255, 0.2);
    font-size: 11px;
    font-weight: 700;
}

/* Order Cards Grid */
.orders-grid {
    display: grid;
    gap: 20px;
}

.order-card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 24px;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 
        6px 6px 12px var(--shadow-dark),
        -6px -6px 12px var(--shadow-light);
    animation: fadeIn 0.5s ease-out;
}

.order-card:hover {
    transform: translateY(-4px);
    box-shadow: 
        10px 10px 20px var(--shadow-dark),
        -10px -10px 20px var(--shadow-light),
        0 0 30px rgba(99, 102, 241, 0.15);
}

.order-card-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 16px;
    padding-bottom: 16px;
    border-bottom: 1px solid var(--border-color);
}

.order-id {
    font-size: 20px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.order-status-badge {
    padding: 8px 16px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: flex;
    align-items: center;
    gap: 6px;
}

.order-status-badge.pending {
    background: rgba(245, 158, 11, 0.15);
    color: #fbbf24;
    border: 1px solid rgba(245, 158, 11, 0.3);
}

.order-status-badge.processing {
    background: rgba(99, 102, 241, 0.15);
    color: var(--primary-light);
    border: 1px solid rgba(99, 102, 241, 0.3);
}

.order-status-badge.success {
    background: rgba(16, 185, 129, 0.15);
    color: #6ee7b7;
    border: 1px solid rgba(16, 185, 129, 0.3);
}

.order-status-badge.failed {
    background: rgba(239, 68, 68, 0.15);
    color: #fca5a5;
    border: 1px solid rgba(239, 68, 68, 0.3);
}

.order-details {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    margin-bottom: 16px;
}

.order-detail-item {
    display: flex;
    flex-direction: column;
    gap: 6px;
}

.order-detail-label {
    font-size: 11px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-weight: 600;
}

.order-detail-value {
    color: var(--text-primary);
    font-weight: 500;
    font-size: 14px;
}

.order-user {
    display: flex;
    align-items: center;
    gap: 10px;
}

.user-avatar {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    background: var(--gradient-bg);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 700;
    font-size: 14px;
    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
}

.user-info {
    display: flex;
    flex-direction: column;
}

.user-name {
    font-weight: 600;
    color: var(--text-primary);
}

.user-username {
    font-size: 12px;
    color: var(--text-secondary);
}

.order-target {
    padding: 12px;
    background: var(--bg-secondary);
    border-radius: 10px;
    font-size: 13px;
    word-break: break-all;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.order-target a {
    color: var(--primary-light);
    text-decoration: none;
}

.order-target a:hover {
    text-decoration: underline;
}

.order-actions {
    display: flex;
    gap: 10px;
    margin-top: 16px;
    padding-top: 16px;
    border-top: 1px solid var(--border-color);
}

.status-select {
    flex: 1;
    padding: 10px 16px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    color: var(--text-primary);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.status-select:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.update-btn {
    padding: 10px 24px;
    background: var(--gradient-bg);
    border: none;
    border-radius: 12px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
}

.update-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(99, 102, 241, 0.4);
}

.completed-badge {
    padding: 10px 16px;
    background: rgba(51, 65, 85, 0.3);
    border-radius: 12px;
    color: var(--text-secondary);
    font-size: 13px;
    text-align: center;
    flex: 1;
}

.order-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid var(--border-color);
    font-size: 12px;
    color: var(--text-secondary);
}

.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: var(--text-secondary);
}

.empty-state-icon {
    font-size: 64px;
    margin-bottom: 16px;
    opacity: 0.5;
}

.empty-state-text {
    font-size: 18px;
    font-weight: 600;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>

<!-- Orders Header -->
<div class="orders-header">
    <div class="orders-title">
        <h1>📦 Order Management</h1>
    </div>
    
    <!-- Filter Tabs -->
    <div class="filter-tabs">
        <a href="orders.php?status=all" class="filter-tab <?php echo $status_filter === 'all' ? 'active' : ''; ?>">
            📋 All Orders
            <span class="filter-count"><?php echo $stats['all']; ?></span>
        </a>
        <a href="orders.php?status=pending" class="filter-tab <?php echo $status_filter === 'pending' ? 'active' : ''; ?>">
            ⏳ Pending
            <span class="filter-count"><?php echo $stats['pending']; ?></span>
        </a>
        <a href="orders.php?status=processing" class="filter-tab <?php echo $status_filter === 'processing' ? 'active' : ''; ?>">
            🔄 Processing
            <span class="filter-count"><?php echo $stats['processing']; ?></span>
        </a>
        <a href="orders.php?status=success" class="filter-tab <?php echo $status_filter === 'success' ? 'active' : ''; ?>">
            ✅ Success
            <span class="filter-count"><?php echo $stats['success']; ?></span>
        </a>
        <a href="orders.php?status=failed" class="filter-tab <?php echo $status_filter === 'failed' ? 'active' : ''; ?>">
            ❌ Failed
            <span class="filter-count"><?php echo $stats['failed']; ?></span>
        </a>
    </div>
</div>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<!-- Orders Grid -->
<div class="orders-grid">
    <?php if (empty($orders)): ?>
        <div class="card">
            <div class="empty-state">
                <div class="empty-state-icon">📭</div>
                <div class="empty-state-text">No orders found</div>
                <p style="margin-top: 8px;">Orders will appear here when users place them</p>
            </div>
        </div>
    <?php else: ?>
        <?php foreach ($orders as $order): ?>
            <div class="order-card">
                <div class="order-card-header">
                    <div class="order-id">#<?php echo $order['id']; ?></div>
                    <div class="order-status-badge <?php echo $order['status']; ?>">
                        <?php
                        $status_icons = [
                            'pending' => '⏳',
                            'processing' => '🔄',
                            'success' => '✅',
                            'failed' => '❌'
                        ];
                        echo $status_icons[$order['status']] ?? '📋';
                        ?>
                        <?php echo ucfirst($order['status']); ?>
                    </div>
                </div>
                
                <div class="order-details">
                    <div class="order-detail-item">
                        <div class="order-detail-label">User</div>
                        <div class="order-user">
                            <div class="user-avatar">
                                <?php echo strtoupper(substr($order['first_name'], 0, 1)); ?>
                            </div>
                            <div class="user-info">
                                <div class="user-name"><?php echo htmlspecialchars($order['first_name']); ?></div>
                                <?php if ($order['username']): ?>
                                    <div class="user-username">@<?php echo htmlspecialchars($order['username']); ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="order-detail-item">
                        <div class="order-detail-label">Service</div>
                        <div class="order-detail-value"><?php echo htmlspecialchars($order['service_name']); ?></div>
                    </div>
                    
                    <div class="order-detail-item">
                        <div class="order-detail-label">Quantity</div>
                        <div class="order-detail-value"><?php echo number_format($order['quantity']); ?></div>
                    </div>
                    
                    <div class="order-detail-item">
                        <div class="order-detail-label">Points Spent</div>
                        <div class="order-detail-value" style="color: var(--primary-light); font-weight: 700;">
                            💎 <?php echo number_format($order['points_spent']); ?>
                        </div>
                    </div>
                </div>
                
                <div class="order-detail-item">
                    <div class="order-detail-label">Target URL</div>
                    <div class="order-target">
                        <a href="<?php echo htmlspecialchars($order['target_url']); ?>" target="_blank">
                            <?php echo htmlspecialchars($order['target_url']); ?>
                        </a>
                    </div>
                </div>
                
                <!-- Actions -->
                <?php if ($order['status'] === 'pending' || $order['status'] === 'processing'): ?>
                    <form method="POST" class="order-actions">
                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                        <input type="hidden" name="admin_note" value="">
                        
                        <select name="status" class="status-select" required>
                            <option value="">Update Status...</option>
                            <option value="processing">🔄 Processing</option>
                            <option value="success">✅ Success</option>
                            <option value="failed">❌ Failed</option>
                        </select>
                        
                        <button type="submit" name="update_status" class="update-btn">
                            Update
                        </button>
                    </form>
                <?php else: ?>
                    <div class="order-actions">
                        <div class="completed-badge">
                            ✓ Order Completed
                        </div>
                    </div>
                <?php endif; ?>
                
                <div class="order-meta">
                    <span>📅 <?php echo date('d M Y, H:i', strtotime($order['created_at'])); ?></span>
                    <span>Order ID: #<?php echo $order['id']; ?></span>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
