<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'Dashboard';

// Get statistics
$db = Database::getInstance()->getConnection();

// Total users
$stmt = $db->query("SELECT COUNT(*) as count FROM users");
$total_users = $stmt->fetch()['count'];

// Total orders
$stmt = $db->query("SELECT COUNT(*) as count FROM orders");
$total_orders = $stmt->fetch()['count'];

// Pending orders
$stmt = $db->query("SELECT COUNT(*) as count FROM orders WHERE status = 'pending'");
$pending_orders = $stmt->fetch()['count'];

// Total points distributed
$stmt = $db->query("SELECT SUM(points) as total FROM users");
$total_points = $stmt->fetch()['total'] ?? 0;

// Success rate
$stmt = $db->query("SELECT COUNT(*) as count FROM orders WHERE status = 'success'");
$success_orders = $stmt->fetch()['count'];
$success_rate = $total_orders > 0 ? round(($success_orders / $total_orders) * 100) : 0;

// Today's stats
$stmt = $db->query("SELECT COUNT(*) as count FROM orders WHERE DATE(created_at) = CURDATE()");
$today_orders = $stmt->fetch()['count'];

$stmt = $db->query("SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = CURDATE()");
$today_users = $stmt->fetch()['count'];

// Recent orders
$stmt = $db->query("
    SELECT o.*, u.username, u.first_name, s.service_name 
    FROM orders o 
    JOIN users u ON o.user_id = u.id 
    JOIN services s ON o.service_id = s.id 
    ORDER BY o.created_at DESC 
    LIMIT 10
");
$recent_orders = $stmt->fetchAll();

// Top services
$stmt = $db->query("
    SELECT s.service_name, COUNT(o.id) as order_count 
    FROM services s 
    LEFT JOIN orders o ON s.id = o.service_id 
    GROUP BY s.id 
    ORDER BY order_count DESC 
    LIMIT 5
");
$top_services = $stmt->fetchAll();

include 'includes/header.php';
?>

<style>
/* Theme Variables */
:root[data-theme="light"] {
    --bg-primary: #f0f4f8;
    --bg-secondary: #ffffff;
    --bg-card: rgba(255, 255, 255, 0.9);
    --text-primary: #1e293b;
    --text-secondary: #64748b;
    --border-color: rgba(0, 0, 0, 0.08);
    --shadow-light: rgba(255, 255, 255, 0.9);
    --shadow-dark: rgba(0, 0, 0, 0.15);
    --gradient-bg: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

:root[data-theme="dark"] {
    --bg-primary: #0f172a;
    --bg-secondary: #1e293b;
    --bg-card: rgba(30, 41, 59, 0.95);
    --text-primary: #f1f5f9;
    --text-secondary: #94a3b8;
    --border-color: rgba(99, 102, 241, 0.1);
    --shadow-light: rgba(255, 255, 255, 0.02);
    --shadow-dark: rgba(0, 0, 0, 0.3);
    --gradient-bg: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

body {
    background: var(--bg-primary);
    transition: background 0.3s ease;
}

/* Theme Toggle */
.theme-toggle {
    position: fixed;
    top: 20px;
    right: 320px;
    z-index: 1000;
}

.theme-switch {
    width: 60px;
    height: 30px;
    background: var(--bg-card);
    border-radius: 15px;
    position: relative;
    cursor: pointer;
    border: 2px solid var(--border-color);
    transition: all 0.3s ease;
    box-shadow: 
        4px 4px 8px var(--shadow-dark),
        -4px -4px 8px var(--shadow-light);
}

.theme-switch::before {
    content: '🌙';
    position: absolute;
    width: 24px;
    height: 24px;
    background: var(--gradient-bg);
    border-radius: 50%;
    top: 1px;
    left: 2px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
}

[data-theme="light"] .theme-switch::before {
    content: '☀️';
    transform: translateX(28px);
}

/* Enhanced Dashboard Header */
.dashboard-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: slideDown 0.6s ease-out;
}

.dashboard-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
}

.dashboard-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

.welcome-text {
    color: var(--text-secondary);
    font-size: 16px;
}

.quick-stats {
    display: flex;
    gap: 24px;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
}

.quick-stat {
    display: flex;
    align-items: center;
    gap: 12px;
}

.quick-stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.quick-stat-info {
    display: flex;
    flex-direction: column;
}

.quick-stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.quick-stat-value {
    font-size: 20px;
    font-weight: 700;
    color: var(--text-primary);
}

/* Enhanced Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 24px;
    margin-bottom: 32px;
}

.stat-card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 28px;
    position: relative;
    overflow: hidden;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    
    /* Neumorphism */
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-bg);
    transform: scaleX(0);
    transform-origin: left;
    transition: transform 0.4s ease;
}

.stat-card:hover::before {
    transform: scaleX(1);
}

.stat-card:hover {
    transform: translateY(-8px) scale(1.02);
    box-shadow: 
        12px 12px 24px var(--shadow-dark),
        -12px -12px 24px var(--shadow-light),
        0 0 40px rgba(99, 102, 241, 0.2);
}

.stat-card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.stat-card-title {
    color: var(--text-secondary);
    font-size: 13px;
    text-transform: uppercase;
    font-weight: 600;
    letter-spacing: 0.5px;
}

.stat-card-icon {
    width: 56px;
    height: 56px;
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 28px;
    box-shadow: 
        inset 4px 4px 8px var(--shadow-dark),
        inset -4px -4px 8px var(--shadow-light);
    animation: float 3s ease-in-out infinite;
}

@keyframes float {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-8px); }
}

.stat-card-value {
    font-size: 42px;
    font-weight: 700;
    margin-bottom: 8px;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    animation: countUp 1s ease-out;
}

@keyframes countUp {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.stat-card-label {
    color: var(--text-secondary);
    font-size: 14px;
}

.stat-card-trend {
    margin-top: 12px;
    padding-top: 12px;
    border-top: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 13px;
}

.trend-up {
    color: #10b981;
}

.trend-down {
    color: #ef4444;
}

/* Enhanced Cards */
.card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px;
    margin-bottom: 24px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: fadeIn 0.6s ease-out;
}

.card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
    padding-bottom: 20px;
    border-bottom: 1px solid var(--border-color);
}

.card-title {
    font-size: 24px;
    font-weight: 600;
    color: var(--text-primary);
}

/* Enhanced Table */
.table-responsive {
    overflow-x: auto;
    border-radius: 16px;
}

table {
    width: 100%;
    border-collapse: collapse;
}

thead {
    background: var(--bg-secondary);
}

th {
    padding: 16px;
    text-align: left;
    font-weight: 600;
    color: var(--text-primary);
    font-size: 13px;
    text-transform: uppercase;
    border-bottom: 2px solid var(--border-color);
    letter-spacing: 0.5px;
}

td {
    padding: 16px;
    border-bottom: 1px solid var(--border-color);
    color: var(--text-primary);
}

tr {
    transition: all 0.3s ease;
}

tr:hover {
    background: var(--bg-secondary);
    transform: scale(1.01);
}

/* Top Services Chart */
.services-chart {
    display: grid;
    gap: 12px;
}

.service-bar {
    display: flex;
    align-items: center;
    gap: 12px;
}

.service-name {
    min-width: 150px;
    font-size: 14px;
    color: var(--text-primary);
    font-weight: 500;
}

.service-progress {
    flex: 1;
    height: 32px;
    background: var(--bg-secondary);
    border-radius: 16px;
    overflow: hidden;
    position: relative;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.service-progress-fill {
    height: 100%;
    background: var(--gradient-bg);
    border-radius: 16px;
    transition: width 1s ease-out;
    display: flex;
    align-items: center;
    justify-content: flex-end;
    padding-right: 12px;
    color: white;
    font-size: 12px;
    font-weight: 600;
}

.service-count {
    min-width: 50px;
    text-align: right;
    font-weight: 700;
    color: var(--text-primary);
}

/* Animations */
@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive */
@media (max-width: 768px) {
    .theme-toggle {
        right: 20px;
    }
    
    .quick-stats {
        flex-direction: column;
        gap: 12px;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<!-- Theme Toggle -->
<div class="theme-toggle">
    <div class="theme-switch" onclick="toggleTheme()"></div>
</div>

<!-- Enhanced Dashboard Header -->
<div class="dashboard-header">
    <div class="dashboard-title">
        <h1>📊 Dashboard</h1>
        <div class="welcome-text">
            Welcome back, <strong><?php echo htmlspecialchars($_SESSION['admin_username']); ?></strong>
        </div>
    </div>
    
    <div class="quick-stats">
        <div class="quick-stat">
            <div class="quick-stat-icon" style="background: rgba(99, 102, 241, 0.1);">📈</div>
            <div class="quick-stat-info">
                <span class="quick-stat-label">Today's Orders</span>
                <span class="quick-stat-value"><?php echo $today_orders; ?></span>
            </div>
        </div>
        <div class="quick-stat">
            <div class="quick-stat-icon" style="background: rgba(16, 185, 129, 0.1);">👤</div>
            <div class="quick-stat-info">
                <span class="quick-stat-label">New Users</span>
                <span class="quick-stat-value"><?php echo $today_users; ?></span>
            </div>
        </div>
        <div class="quick-stat">
            <div class="quick-stat-icon" style="background: rgba(245, 158, 11, 0.1);">✅</div>
            <div class="quick-stat-info">
                <span class="quick-stat-label">Success Rate</span>
                <span class="quick-stat-value"><?php echo $success_rate; ?>%</span>
            </div>
        </div>
    </div>
</div>

<!-- Stats Grid -->
<div class="stats-grid">
    <div class="stat-card">
        <div class="stat-card-header">
            <span class="stat-card-title">Total Users</span>
            <div class="stat-card-icon" style="background: rgba(99, 102, 241, 0.1); color: var(--primary);">
                👥
            </div>
        </div>
        <div class="stat-card-value"><?php echo number_format($total_users); ?></div>
        <div class="stat-card-label">Registered users</div>
        <div class="stat-card-trend trend-up">
            ↗ +<?php echo $today_users; ?> today
        </div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-header">
            <span class="stat-card-title">Total Orders</span>
            <div class="stat-card-icon" style="background: rgba(139, 92, 246, 0.1); color: var(--secondary);">
                📦
            </div>
        </div>
        <div class="stat-card-value"><?php echo number_format($total_orders); ?></div>
        <div class="stat-card-label">All time orders</div>
        <div class="stat-card-trend trend-up">
            ↗ +<?php echo $today_orders; ?> today
        </div>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-header">
            <span class="stat-card-title">Pending Orders</span>
            <div class="stat-card-icon" style="background: rgba(245, 158, 11, 0.1); color: var(--warning);">
                ⏳
            </div>
        </div>
        <div class="stat-card-value"><?php echo number_format($pending_orders); ?></div>
        <div class="stat-card-label">Awaiting action</div>
        <?php if ($pending_orders > 0): ?>
            <div class="stat-card-trend" style="color: var(--warning);">
                ⚠ Needs attention
            </div>
        <?php else: ?>
            <div class="stat-card-trend trend-up">
                ✓ All clear
            </div>
        <?php endif; ?>
    </div>
    
    <div class="stat-card">
        <div class="stat-card-header">
            <span class="stat-card-title">Total Points</span>
            <div class="stat-card-icon" style="background: rgba(16, 185, 129, 0.1); color: var(--success);">
                💎
            </div>
        </div>
        <div class="stat-card-value"><?php echo number_format($total_points); ?></div>
        <div class="stat-card-label">Points in circulation</div>
        <div class="stat-card-trend trend-up">
            ↗ Active economy
        </div>
    </div>
</div>

<!-- Top Services -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">📊 Top Services</h2>
    </div>
    
    <div class="services-chart">
        <?php 
        $max_orders = $top_services[0]['order_count'] ?? 1;
        foreach ($top_services as $service): 
            $percentage = $max_orders > 0 ? ($service['order_count'] / $max_orders) * 100 : 0;
        ?>
            <div class="service-bar">
                <div class="service-name"><?php echo htmlspecialchars($service['service_name']); ?></div>
                <div class="service-progress">
                    <div class="service-progress-fill" style="width: <?php echo $percentage; ?>%">
                        <?php if ($percentage > 20): ?>
                            <?php echo $service['order_count']; ?> orders
                        <?php endif; ?>
                    </div>
                </div>
                <div class="service-count"><?php echo $service['order_count']; ?></div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Recent Orders -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">📋 Recent Orders</h2>
        <a href="orders.php" class="btn btn-primary btn-sm">View All</a>
    </div>
    
    <div class="table-responsive">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>User</th>
                    <th>Service</th>
                    <th>Target</th>
                    <th>Points</th>
                    <th>Status</th>
                    <th>Date</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($recent_orders)): ?>
                    <tr>
                        <td colspan="7" style="text-align: center; color: var(--text-secondary);">No orders yet</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($recent_orders as $order): ?>
                        <tr>
                            <td><strong>#<?php echo $order['id']; ?></strong></td>
                            <td>
                                <?php echo htmlspecialchars($order['first_name']); ?>
                                <?php if ($order['username']): ?>
                                    <br><small style="color: var(--text-secondary);">@<?php echo htmlspecialchars($order['username']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($order['service_name']); ?></td>
                            <td>
                                <a href="<?php echo htmlspecialchars($order['target_url']); ?>" target="_blank" style="color: var(--primary); text-decoration: none;">
                                    <?php echo substr($order['target_url'], 0, 40) . '...'; ?>
                                </a>
                            </td>
                            <td><strong><?php echo number_format($order['points_spent']); ?></strong></td>
                            <td>
                                <span class="badge badge-<?php echo $order['status']; ?>">
                                    <?php echo ucfirst($order['status']); ?>
                                </span>
                            </td>
                            <td><?php echo date('d M Y, H:i', strtotime($order['created_at'])); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
// Theme Toggle
function toggleTheme() {
    const root = document.documentElement;
    const currentTheme = root.getAttribute('data-theme') || 'dark';
    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
    
    root.setAttribute('data-theme', newTheme);
    localStorage.setItem('theme', newTheme);
}

// Load saved theme
document.addEventListener('DOMContentLoaded', function() {
    const savedTheme = localStorage.getItem('theme') || 'dark';
    document.documentElement.setAttribute('data-theme', savedTheme);
    
    // Animate progress bars
    setTimeout(() => {
        document.querySelectorAll('.service-progress-fill').forEach(bar => {
            const width = bar.style.width;
            bar.style.width = '0%';
            setTimeout(() => {
                bar.style.width = width;
            }, 100);
        });
    }, 500);
});
</script>

<?php include 'includes/footer.php'; ?>
