<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'Force Join Channels';
$db = Database::getInstance()->getConnection();

$success = '';
$error = '';

// Handle channel add
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_channel'])) {
    $channel_id = trim($_POST['channel_id']);
    $channel_link = trim($_POST['channel_link']);
    $channel_name = trim($_POST['channel_name']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    if (empty($channel_id) || empty($channel_link)) {
        $error = "Channel ID and Link are required!";
    } else {
        try {
            $stmt = $db->prepare("INSERT INTO force_join_channels (channel_id, channel_link, channel_name, is_active) VALUES (?, ?, ?, ?)");
            $stmt->execute([$channel_id, $channel_link, $channel_name, $is_active]);
            $success = "Channel added successfully!";
        } catch(PDOException $e) {
            if ($e->getCode() == 23000) {
                $error = "Channel ID already exists!";
            } else {
                $error = "Error adding channel: " . $e->getMessage();
            }
        }
    }
}

// Handle channel update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_channel'])) {
    $channel_id = $_POST['id'];
    $channel_link = trim($_POST['channel_link']);
    $channel_name = trim($_POST['channel_name']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    $stmt = $db->prepare("UPDATE force_join_channels SET channel_link = ?, channel_name = ?, is_active = ? WHERE id = ?");
    if ($stmt->execute([$channel_link, $channel_name, $is_active, $channel_id])) {
        $success = "Channel updated successfully!";
    }
}

// Handle channel delete
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $channel_id = $_GET['delete'];
    
    try {
        $stmt = $db->prepare("DELETE FROM force_join_channels WHERE id = ?");
        $stmt->execute([$channel_id]);
        $success = "Channel deleted successfully!";
    } catch(PDOException $e) {
        $error = "Error deleting channel: " . $e->getMessage();
    }
}

// Get all channels
$stmt = $db->query("SELECT * FROM force_join_channels ORDER BY id DESC");
$channels = $stmt->fetchAll();

include 'includes/header.php';
?>

<style>
/* Channels Page Specific Styles */
.channels-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: slideDown 0.6s ease-out;
}

.channels-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.channels-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

.add-channel-btn {
    padding: 12px 24px;
    background: var(--gradient-bg);
    border: none;
    border-radius: 12px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
}

.add-channel-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(99, 102, 241, 0.4);
}

.channels-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 24px;
}

.channel-card {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 24px;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    animation: fadeIn 0.6s ease-out;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.channel-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-bg);
    transform: scaleX(0);
    transition: transform 0.4s ease;
}

.channel-card:hover::before {
    transform: scaleX(1);
}

.channel-card:hover {
    transform: translateY(-8px);
    box-shadow: 
        12px 12px 24px var(--shadow-dark),
        -12px -12px 24px var(--shadow-light),
        0 0 40px rgba(99, 102, 241, 0.2);
}

.channel-card.inactive {
    opacity: 0.6;
}

.channel-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
}

.channel-icon {
    font-size: 40px;
    width: 70px;
    height: 70px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: rgba(99, 102, 241, 0.1);
    border-radius: 16px;
    animation: float 3s ease-in-out infinite;
    box-shadow: 
        inset 4px 4px 8px var(--shadow-dark),
        inset -4px -4px 8px var(--shadow-light);
}

@keyframes float {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-8px); }
}

.channel-status {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
}

.channel-status.active {
    background: rgba(16, 185, 129, 0.15);
    color: #6ee7b7;
    border: 1px solid rgba(16, 185, 129, 0.3);
}

.channel-status.inactive {
    background: rgba(148, 163, 184, 0.15);
    color: var(--text-secondary);
    border: 1px solid rgba(148, 163, 184, 0.3);
}

.channel-name {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 6px;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.channel-id {
    color: var(--text-secondary);
    font-size: 13px;
    margin-bottom: 12px;
    font-family: 'Courier New', monospace;
}

.channel-link-box {
    padding: 12px;
    background: var(--bg-secondary);
    border-radius: 10px;
    margin-bottom: 16px;
    word-break: break-all;
    box-shadow: 
        inset 2px 2px 4px var(--shadow-dark),
        inset -2px -2px 4px var(--shadow-light);
}

.channel-link-box a {
    color: var(--primary-light);
    text-decoration: none;
    font-size: 13px;
}

.channel-link-box a:hover {
    text-decoration: underline;
}

.channel-actions {
    display: flex;
    gap: 10px;
}

.channel-action-btn {
    flex: 1;
    padding: 10px;
    border: none;
    border-radius: 10px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 
        4px 4px 8px var(--shadow-dark),
        -4px -4px 8px var(--shadow-light);
}

.channel-action-btn.edit {
    background: rgba(59, 130, 246, 0.2);
    color: #93c5fd;
    border: 1px solid rgba(59, 130, 246, 0.3);
}

.channel-action-btn.edit:hover {
    background: rgba(59, 130, 246, 0.3);
    transform: translateY(-2px);
}

.channel-action-btn.delete {
    background: rgba(239, 68, 68, 0.2);
    color: #fca5a5;
    border: 1px solid rgba(239, 68, 68, 0.3);
}

.channel-action-btn.delete:hover {
    background: rgba(239, 68, 68, 0.3);
    transform: translateY(-2px);
}

.edit-form {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>

<!-- Channels Header -->
<div class="channels-header">
    <div class="channels-title">
        <h1>📢 Force Join Channels</h1>
        <button class="add-channel-btn" onclick="toggleAddForm()">
            ➕ Add Channel
        </button>
    </div>
</div>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<!-- Add Channel Form -->
<div class="card" id="addChannelForm" style="display: none; margin-bottom: 30px;">
    <div class="card-header">
        <h2 class="card-title">➕ Add New Channel</h2>
        <button class="btn-close" onclick="toggleAddForm()" style="background: none; border: none; color: var(--text-secondary); font-size: 24px; cursor: pointer;">✕</button>
    </div>
    
    <form method="POST" style="padding: 0 10px 20px;">
        <div class="form-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 20px;">
            <div class="form-group-modern">
                <label for="channel_id">Channel ID</label>
                <input type="text" id="channel_id" name="channel_id" placeholder="e.g., -1002497131761" required
                       style="width: 100%; padding: 12px 16px; background: var(--bg-secondary); border: 2px solid var(--border-color); border-radius: 12px; color: var(--text-primary);">
                <small style="color: var(--text-secondary); font-size: 12px;">Get from channel info (starts with -100)</small>
            </div>
            
            <div class="form-group-modern">
                <label for="channel_link">Channel Link</label>
                <input type="text" id="channel_link" name="channel_link" placeholder="https://t.me/+..." required
                       style="width: 100%; padding: 12px 16px; background: var(--bg-secondary); border: 2px solid var(--border-color); border-radius: 12px; color: var(--text-primary);">
                <small style="color: var(--text-secondary); font-size: 12px;">Full channel invite link</small>
            </div>
            
            <div class="form-group-modern">
                <label for="channel_name">Channel Name (Optional)</label>
                <input type="text" id="channel_name" name="channel_name" placeholder="e.g., My Channel"
                       style="width: 100%; padding: 12px 16px; background: var(--bg-secondary); border: 2px solid var(--border-color); border-radius: 12px; color: var(--text-primary);">
            </div>
        </div>
        
        <div class="form-group-modern" style="margin-bottom: 20px;">
            <label class="toggle-switch" style="display: flex; align-items: center; gap: 12px; cursor: pointer;">
                <input type="checkbox" name="is_active" checked style="display: none;">
                <span class="toggle-slider" style="position: relative; width: 50px; height: 26px; background: rgba(51, 65, 85, 0.6); border-radius: 13px; transition: all 0.3s ease;"></span>
                <span class="toggle-label">Active (Force users to join)</span>
            </label>
        </div>
        
        <div style="display: flex; gap: 12px;">
            <button type="submit" name="add_channel" class="btn btn-primary">
                ✨ Add Channel
            </button>
            <button type="button" class="btn btn-secondary" onclick="toggleAddForm()">
                Cancel
            </button>
        </div>
    </form>
</div>

<!-- Channels Grid -->
<div class="channels-grid">
    <?php if (empty($channels)): ?>
        <div class="card">
            <div class="empty-state" style="text-align: center; padding: 60px 20px;">
                <div style="font-size: 64px; margin-bottom: 16px; opacity: 0.5;">📢</div>
                <div style="font-size: 18px; font-weight: 600; color: var(--text-primary);">No channels added</div>
                <p style="margin-top: 8px; color: var(--text-secondary);">Add channels to force users to join before using the bot</p>
            </div>
        </div>
    <?php else: ?>
        <?php foreach ($channels as $channel): ?>
            <div class="channel-card <?php echo $channel['is_active'] ? 'active' : 'inactive'; ?>">
                <div class="channel-header">
                    <div class="channel-icon">📢</div>
                    <div class="channel-status <?php echo $channel['is_active'] ? 'active' : 'inactive'; ?>">
                        <?php echo $channel['is_active'] ? '● Active' : '● Inactive'; ?>
                    </div>
                </div>
                
                <h3 class="channel-name"><?php echo htmlspecialchars($channel['channel_name'] ?: 'Channel ' . $channel['id']); ?></h3>
                <p class="channel-id">ID: <?php echo htmlspecialchars($channel['channel_id']); ?></p>
                
                <div class="channel-link-box">
                    <a href="<?php echo htmlspecialchars($channel['channel_link']); ?>" target="_blank">
                        <?php echo htmlspecialchars($channel['channel_link']); ?>
                    </a>
                </div>
                
                <div class="channel-actions">
                    <button class="channel-action-btn edit" onclick="editChannel(<?php echo $channel['id']; ?>)">
                        ✏️ Edit
                    </button>
                    <button class="channel-action-btn delete" onclick="deleteChannel(<?php echo $channel['id']; ?>, '<?php echo htmlspecialchars($channel['channel_name'] ?: 'this channel'); ?>')">
                        🗑️ Delete
                    </button>
                </div>
                
                <!-- Edit Form (Hidden) -->
                <div class="edit-form" id="editForm<?php echo $channel['id']; ?>" style="display: none;">
                    <form method="POST">
                        <input type="hidden" name="id" value="<?php echo $channel['id']; ?>">
                        
                        <div class="form-group-modern" style="margin-bottom: 16px;">
                            <label>Channel Link</label>
                            <input type="text" name="channel_link" value="<?php echo htmlspecialchars($channel['channel_link']); ?>" required
                                   style="width: 100%; padding: 10px 14px; background: var(--bg-secondary); border: 2px solid var(--border-color); border-radius: 10px; color: var(--text-primary);">
                        </div>
                        
                        <div class="form-group-modern" style="margin-bottom: 16px;">
                            <label>Channel Name</label>
                            <input type="text" name="channel_name" value="<?php echo htmlspecialchars($channel['channel_name']); ?>"
                                   style="width: 100%; padding: 10px 14px; background: var(--bg-secondary); border: 2px solid var(--border-color); border-radius: 10px; color: var(--text-primary);">
                        </div>
                        
                        <div class="form-group-modern" style="margin-bottom: 16px;">
                            <label class="toggle-switch" style="display: flex; align-items: center; gap: 12px; cursor: pointer;">
                                <input type="checkbox" name="is_active" <?php echo $channel['is_active'] ? 'checked' : ''; ?> style="display: none;">
                                <span class="toggle-slider"></span>
                                <span class="toggle-label">Active</span>
                            </label>
                        </div>
                        
                        <div style="display: flex; gap: 8px;">
                            <button type="submit" name="update_channel" class="btn btn-success btn-sm">
                                💾 Save
                            </button>
                            <button type="button" class="btn btn-secondary btn-sm" onclick="cancelEdit(<?php echo $channel['id']; ?>)">
                                Cancel
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
function toggleAddForm() {
    const form = document.getElementById('addChannelForm');
    if (form.style.display === 'none') {
        form.style.display = 'block';
        form.scrollIntoView({ behavior: 'smooth' });
    } else {
        form.style.display = 'none';
    }
}

function editChannel(id) {
    const editForm = document.getElementById('editForm' + id);
    const allForms = document.querySelectorAll('.edit-form');
    
    allForms.forEach(form => {
        if (form.id !== 'editForm' + id) {
            form.style.display = 'none';
        }
    });
    
    if (editForm.style.display === 'none') {
        editForm.style.display = 'block';
    } else {
        editForm.style.display = 'none';
    }
}

function cancelEdit(id) {
    document.getElementById('editForm' + id).style.display = 'none';
}

function deleteChannel(id, name) {
    if (confirm('Are you sure you want to delete "' + name + '"?\n\nThis action cannot be undone!')) {
        window.location.href = 'channels.php?delete=' + id;
    }
}

// Toggle switch functionality
document.querySelectorAll('.toggle-switch input[type="checkbox"]').forEach(checkbox => {
    const slider = checkbox.nextElementSibling;
    
    checkbox.addEventListener('change', function() {
        if (this.checked) {
            slider.style.background = 'linear-gradient(135deg, #43e97b 0%, #38f9d7 100%)';
        } else {
            slider.style.background = 'rgba(51, 65, 85, 0.6)';
        }
    });
    
    // Initialize
    if (checkbox.checked) {
        slider.style.background = 'linear-gradient(135deg, #43e97b 0%, #38f9d7 100%)';
    }
});
</script>

<style>
.toggle-slider::before {
    content: '';
    position: absolute;
    width: 22px;
    height: 22px;
    border-radius: 50%;
    background: white;
    top: 2px;
    left: 2px;
    transition: all 0.3s ease;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
}

.toggle-switch input:checked + .toggle-slider::before {
    transform: translateX(24px);
}
</style>

<?php include 'includes/footer.php'; ?>
