<?php
require_once 'auth.php';
requireAdmin();

$page_title = 'Broadcast Message';
$db = Database::getInstance()->getConnection();

$success = '';
$error = '';
$broadcast_sent = false;

// Handle broadcast
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_broadcast'])) {
    $message = trim($_POST['message']);
    $target = $_POST['target'] ?? 'all';
    $min_points = intval($_POST['min_points'] ?? 0);
    
    if (empty($message)) {
        $error = "Message cannot be empty!";
    } else {
        require_once __DIR__ . '/../bot/telegram.php';
        
        // Build query based on target
        $query = "SELECT telegram_id, first_name FROM users WHERE 1=1";
        $params = [];
        
        if ($target === 'active') {
            $query .= " AND (SELECT COUNT(*) FROM orders WHERE user_id = users.id) > 0";
        } elseif ($target === 'inactive') {
            $query .= " AND (SELECT COUNT(*) FROM orders WHERE user_id = users.id) = 0";
        } elseif ($target === 'min_points') {
            $query .= " AND points >= ?";
            $params[] = $min_points;
        }
        
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        $users = $stmt->fetchAll();
        
        $sent_count = 0;
        $failed_count = 0;
        
        foreach ($users as $user) {
            $formatted_message = "📢 <b>Broadcast Message</b>\n\n" . $message;
            
            try {
                $result = sendMessage($user['telegram_id'], $formatted_message);
                if ($result) {
                    $sent_count++;
                } else {
                    $failed_count++;
                }
                
                // Small delay to avoid rate limiting
                usleep(50000); // 50ms delay
            } catch (Exception $e) {
                $failed_count++;
            }
        }
        
        $success = "Broadcast sent to {$sent_count} users successfully!";
        if ($failed_count > 0) {
            $success .= " ({$failed_count} failed)";
        }
        $broadcast_sent = true;
    }
}

// Get user statistics
$stmt = $db->query("SELECT COUNT(*) as total FROM users");
$total_users = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as active FROM users WHERE (SELECT COUNT(*) FROM orders WHERE user_id = users.id) > 0");
$active_users = $stmt->fetch()['active'];

$inactive_users = $total_users - $active_users;

include 'includes/header.php';
?>

<style>
/* Broadcast Page Specific Styles */
.broadcast-header {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 24px;
    padding: 32px 40px;
    margin-bottom: 32px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    animation: slideDown 0.6s ease-out;
}

.broadcast-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.broadcast-title h1 {
    font-size: 36px;
    font-weight: 700;
    background: var(--gradient-bg);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin: 0;
}

.broadcast-stats {
    display: flex;
    gap: 24px;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
}

.broadcast-stat {
    display: flex;
    align-items: center;
    gap: 12px;
}

.broadcast-stat-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.broadcast-stat-info {
    display: flex;
    flex-direction: column;
}

.broadcast-stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
}

.broadcast-stat-value {
    font-size: 24px;
    font-weight: 700;
    color: var(--text-primary);
}

.broadcast-container {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 24px;
}

.broadcast-composer {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 28px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
}

.composer-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    padding-bottom: 16px;
    border-bottom: 1px solid var(--border-color);
}

.form-group-broadcast {
    margin-bottom: 24px;
}

.form-group-broadcast label {
    display: block;
    margin-bottom: 10px;
    color: var(--text-primary);
    font-weight: 600;
    font-size: 14px;
}

.message-textarea {
    width: 100%;
    min-height: 200px;
    padding: 16px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    color: var(--text-primary);
    font-size: 15px;
    font-family: inherit;
    resize: vertical;
    transition: all 0.3s ease;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.message-textarea:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light),
        0 0 0 3px rgba(99, 102, 241, 0.1);
}

.target-options {
    display: grid;
    gap: 12px;
}

.target-option {
    display: flex;
    align-items: center;
    padding: 14px 18px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.target-option:hover {
    border-color: var(--primary);
    background: rgba(99, 102, 241, 0.05);
}

.target-option input[type="radio"] {
    margin-right: 12px;
    width: 20px;
    height: 20px;
    cursor: pointer;
}

.target-option-content {
    flex: 1;
}

.target-option-title {
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 4px;
}

.target-option-desc {
    font-size: 12px;
    color: var(--text-secondary);
}

.target-option-count {
    padding: 4px 12px;
    background: var(--gradient-bg);
    color: white;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 700;
}

.min-points-input {
    width: 100%;
    padding: 12px 16px;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 12px;
    color: var(--text-primary);
    font-size: 15px;
    margin-top: 12px;
    transition: all 0.3s ease;
}

.min-points-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.send-broadcast-btn {
    width: 100%;
    padding: 16px;
    background: var(--gradient-bg);
    border: none;
    border-radius: 12px;
    color: white;
    font-size: 16px;
    font-weight: 700;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 8px 20px rgba(99, 102, 241, 0.3);
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
}

.send-broadcast-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 12px 30px rgba(99, 102, 241, 0.4);
}

.send-broadcast-btn:active {
    transform: translateY(0);
}

.broadcast-preview {
    background: var(--bg-card);
    backdrop-filter: blur(20px);
    border: 1px solid var(--border-color);
    border-radius: 20px;
    padding: 28px;
    box-shadow: 
        8px 8px 16px var(--shadow-dark),
        -8px -8px 16px var(--shadow-light);
    position: sticky;
    top: 20px;
}

.preview-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    padding-bottom: 16px;
    border-bottom: 1px solid var(--border-color);
}

.preview-phone {
    background: var(--bg-secondary);
    border-radius: 16px;
    padding: 20px;
    box-shadow: 
        inset 3px 3px 6px var(--shadow-dark),
        inset -3px -3px 6px var(--shadow-light);
}

.preview-message {
    background: rgba(99, 102, 241, 0.1);
    border-left: 4px solid var(--primary);
    padding: 16px;
    border-radius: 8px;
    color: var(--text-primary);
    font-size: 14px;
    line-height: 1.6;
    white-space: pre-wrap;
    word-wrap: break-word;
}

.preview-empty {
    text-align: center;
    padding: 40px 20px;
    color: var(--text-secondary);
    font-size: 14px;
}

.formatting-tips {
    margin-top: 20px;
    padding: 16px;
    background: rgba(99, 102, 241, 0.05);
    border-radius: 12px;
    border: 1px solid rgba(99, 102, 241, 0.1);
}

.formatting-tips-title {
    font-size: 13px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.formatting-tips-list {
    font-size: 12px;
    color: var(--text-secondary);
    line-height: 1.8;
}

.formatting-tips-list code {
    background: var(--bg-secondary);
    padding: 2px 6px;
    border-radius: 4px;
    font-family: 'Courier New', monospace;
    color: var(--primary-light);
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@media (max-width: 968px) {
    .broadcast-container {
        grid-template-columns: 1fr;
    }
    
    .broadcast-preview {
        position: relative;
        top: 0;
    }
}
</style>

<!-- Broadcast Header -->
<div class="broadcast-header">
    <div class="broadcast-title">
        <h1>📢 Broadcast Message</h1>
    </div>
    
    <div class="broadcast-stats">
        <div class="broadcast-stat">
            <div class="broadcast-stat-icon" style="background: rgba(99, 102, 241, 0.1);">👥</div>
            <div class="broadcast-stat-info">
                <span class="broadcast-stat-label">Total Users</span>
                <span class="broadcast-stat-value"><?php echo $total_users; ?></span>
            </div>
        </div>
        <div class="broadcast-stat">
            <div class="broadcast-stat-icon" style="background: rgba(16, 185, 129, 0.1);">✅</div>
            <div class="broadcast-stat-info">
                <span class="broadcast-stat-label">Active Users</span>
                <span class="broadcast-stat-value"><?php echo $active_users; ?></span>
            </div>
        </div>
        <div class="broadcast-stat">
            <div class="broadcast-stat-icon" style="background: rgba(245, 158, 11, 0.1);">💤</div>
            <div class="broadcast-stat-info">
                <span class="broadcast-stat-label">Inactive Users</span>
                <span class="broadcast-stat-value"><?php echo $inactive_users; ?></span>
            </div>
        </div>
    </div>
</div>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<div class="broadcast-container">
    <!-- Message Composer -->
    <div class="broadcast-composer">
        <h2 class="composer-title">✍️ Compose Message</h2>
        
        <form method="POST" id="broadcastForm">
            <div class="form-group-broadcast">
                <label for="message">Message Content</label>
                <textarea 
                    name="message" 
                    id="message" 
                    class="message-textarea" 
                    placeholder="Type your broadcast message here...&#10;&#10;You can use HTML formatting:&#10;- <b>Bold text</b>&#10;- <i>Italic text</i>&#10;- <code>Code</code>"
                    required
                    oninput="updatePreview()"
                ><?php echo isset($_POST['message']) ? htmlspecialchars($_POST['message']) : ''; ?></textarea>
            </div>
            
            <div class="form-group-broadcast">
                <label>Target Audience</label>
                <div class="target-options">
                    <label class="target-option">
                        <input type="radio" name="target" value="all" checked onchange="toggleMinPoints()">
                        <div class="target-option-content">
                            <div class="target-option-title">All Users</div>
                            <div class="target-option-desc">Send to everyone</div>
                        </div>
                        <span class="target-option-count"><?php echo $total_users; ?></span>
                    </label>
                    
                    <label class="target-option">
                        <input type="radio" name="target" value="active" onchange="toggleMinPoints()">
                        <div class="target-option-content">
                            <div class="target-option-title">Active Users</div>
                            <div class="target-option-desc">Users who have placed orders</div>
                        </div>
                        <span class="target-option-count"><?php echo $active_users; ?></span>
                    </label>
                    
                    <label class="target-option">
                        <input type="radio" name="target" value="inactive" onchange="toggleMinPoints()">
                        <div class="target-option-content">
                            <div class="target-option-title">Inactive Users</div>
                            <div class="target-option-desc">Users who haven't ordered yet</div>
                        </div>
                        <span class="target-option-count"><?php echo $inactive_users; ?></span>
                    </label>
                    
                    <label class="target-option">
                        <input type="radio" name="target" value="min_points" onchange="toggleMinPoints()">
                        <div class="target-option-content">
                            <div class="target-option-title">By Points</div>
                            <div class="target-option-desc">Users with minimum points</div>
                        </div>
                    </label>
                </div>
                
                <div id="minPointsContainer" style="display: none;">
                    <input 
                        type="number" 
                        name="min_points" 
                        class="min-points-input" 
                        placeholder="Minimum points required"
                        min="0"
                        value="100"
                    >
                </div>
            </div>
            
            <button type="submit" name="send_broadcast" class="send-broadcast-btn">
                <span>📤</span>
                <span>Send Broadcast</span>
            </button>
        </form>
    </div>
    
    <!-- Preview Panel -->
    <div class="broadcast-preview">
        <h3 class="preview-title">📱 Preview</h3>
        
        <div class="preview-phone">
            <div id="previewContent" class="preview-empty">
                Type a message to see preview...
            </div>
        </div>
        
        <div class="formatting-tips">
            <div class="formatting-tips-title">💡 Formatting Tips</div>
            <div class="formatting-tips-list">
                • <code>&lt;b&gt;text&lt;/b&gt;</code> - Bold<br>
                • <code>&lt;i&gt;text&lt;/i&gt;</code> - Italic<br>
                • <code>&lt;code&gt;text&lt;/code&gt;</code> - Code<br>
                • <code>&lt;a href="url"&gt;text&lt;/a&gt;</code> - Link
            </div>
        </div>
    </div>
</div>

<script>
function updatePreview() {
    const message = document.getElementById('message').value;
    const preview = document.getElementById('previewContent');
    
    if (message.trim() === '') {
        preview.className = 'preview-empty';
        preview.textContent = 'Type a message to see preview...';
    } else {
        preview.className = 'preview-message';
        preview.innerHTML = '📢 <b>Broadcast Message</b><br><br>' + message;
    }
}

function toggleMinPoints() {
    const minPointsRadio = document.querySelector('input[name="target"][value="min_points"]');
    const minPointsContainer = document.getElementById('minPointsContainer');
    
    if (minPointsRadio.checked) {
        minPointsContainer.style.display = 'block';
    } else {
        minPointsContainer.style.display = 'none';
    }
}

// Confirm before sending
document.getElementById('broadcastForm').addEventListener('submit', function(e) {
    const target = document.querySelector('input[name="target"]:checked').value;
    const targetLabels = {
        'all': 'ALL users',
        'active': 'ACTIVE users',
        'inactive': 'INACTIVE users',
        'min_points': 'users with minimum points'
    };
    
    if (!confirm(`Are you sure you want to send this broadcast to ${targetLabels[target]}?\n\nThis action cannot be undone!`)) {
        e.preventDefault();
    }
});

// Initialize preview
updatePreview();
</script>

<?php include 'includes/footer.php'; ?>
